--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Command Handler
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

COMMAND = {}

function ExecuteCommand(strCommand, tParams)
	trace('ExecuteCommand', strCommand)
	tParams = tParams or {}
	local strCmd = string.gsub(strCommand, " ","")
	if (strCommand == 'LUA_ACTION') then
		for cmd,cmdv in pairs(tParams) do
			if (cmd == 'ACTION') then
				if (COMMAND[cmdv] ~= nil and type(COMMAND[cmdv]) == 'function') then
					COMMAND[cmdv](tParams)
				else
					dbg('Unknown Action Command.')
				end
			end
		end
	else
		if (COMMAND[strCmd] ~= nil and type(COMMAND[strCmd]) == 'function') then
			COMMAND[strCmd](tParams)
		else
			dbg('Unknown command.')
		end
	end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Debugging
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function dbg(...)
	if (Properties['DEBUG MODE'] == 'ON') then print(...) end
end

function trace(strFn, ...)
	if (Properties['DEBUG MODE'] ~= 'ON') then return end
	for k,v in pairs(arg) do arg[k] = tostring(v) end
	dbg(string.format('[%s](%s)', strFn, table.concat(arg, ', ')))
end

function err(...)
	if (Properties['DEBUG MODE'] ~= 'ON') then return end
	for k,v in pairs(arg) do arg[k] = tostring(v) end
	dbg(string.format('[ERROR]: %s', table.concat(arg, ', ')))
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--HOUSELOGIX EXCEPTION HANDLING ROUTINES
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function NilException(...)
    local throw = false
    local msg = ""
    for i=1,select("#",...) do
        local temp = select(i,...)
        if (temp) then
            if (i == 1) then
                temp = string.format("(%s): ", temp)
            elseif (type(temp) == "string") then
                temp = string.format("\"%s\"",temp)
            end
            msg = msg .. temp .. " "
        else
            msg = msg .. "NIL PARAM"
            throw = true
        end
    end
    if (throw) then msg = msg .. " -- THROWING AWAY" end
    dbg(msg)
    return throw
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Properties handler
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

PROPS = {}

function OnPropertyChanged(strProperty)
	trace('OnPropertyChanged', strProperty)
	local strProp = string.gsub(strProperty, ' ','')
	if (PROPS[strProp] ~= nil and type(PROPS[strProp]) == 'function') then
		PROPS[strProp](Properties[strProperty])
	end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Lua table utils
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function PrintTable(tValue, sIndent)
	sIndent = sIndent or '   '
	for k,v in pairs(tValue) do
		print(sIndent .. tostring(k) .. ':  ' .. tostring(v))
		if (type(v) == 'table') then
			PrintTable(v, sIndent .. '   ')
		end
	end
end

function AlphabeticalPairs(t, f)
	local a = {}

	for n in pairs(t) do
		table.insert(a, n)
	end

	table.sort(a, f)

	local index = 0      -- iterator variable
	local iter = function ()   -- iterator function
			index = index + 1
			if a[index] == nil then
				return nil
			else
				return a[index], t[a[index]]
			end
		end

	return iter
end

-- Be aware, this is really a hack. You could get the wrong key for a value,
-- especially if your data has duplicate values.
function AlphabeticalPairsByValue(t, f)
	local a = {}

	for k,v in pairs(t) do
		table.insert(a, v)
	end

	table.sort(a, f)

	local index = 0
	local iter = function()
			index = index + 1
			if a[index] == nil then
				return nil
			else
				for k,v in pairs(t) do
					if v == a[index] then
						return k, v
					end
				end
			end
		end

	return iter
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Timer Helpers
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

if not HOUSELOGIX then HOUSELOGIX = {} end

function OnTimerExpired(idTimer)
	local timer = pullTimer(idTimer)
	if (not timer) then dbg("UNKNOWN TIMER FIRED | THROWING AWAY") return end
	timer.handler()
end

--------------------------------
-- Helpers
--------------------------------
g_timers = {}

function startTimer(value, units, callback)
	local timerId = C4:AddTimer(tonumber(value), units)
	g_timers[timerId] = { handler = callback }
	return timerId
end

function pullTimer(timerId)
	local timer = g_timers[timerId]
	g_timers[timerId] = nil
	return timer
end

function killTimer(timerId)
	local timer = g_timers[timerId]
	g_timers[timerId] = nil
	if tonumber(timerId) > 0 then
		C4:KillTimer(timerId)
	end
	return timer
end

--------------------------------
-- Classes
--------------------------------

--------------------------------
-- Resets self if started again before expiration
--------------------------------
function HOUSELOGIX.SingletonTimer(nInterval, strUnits, fnCallback)
	local self = {}

	local ID = 0
	local interval = nInterval
	local units = strUnits
	local callback = fnCallback

	function self.Start()
		if ID ~= 0 then
			killTimer(ID)
		end

		ID = startTimer(interval, units, callback)
	end

	function self.Stop()
		if ID ~= 0 then
			killTimer(ID)
			ID = 0
		end
	end

	return self
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- URL Callbacks
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

g_responseTickets = {}

function ReceivedAsync(ticketId, strData, responseCode, tHeaders, strError)
	local ticket = pullUrlTicket(ticketId)
	if (not ticket) then dbg("[ReceivedAsync]: Unknown Ticket | Throwing Away") return end

	if (strError) then
		if (ticket.errorHandler and type(ticket.errorHandler) == 'function') then
			ticket.errorHandler(strError)
		else
			dbg("[ReceivedAsync]: ERROR | " .. strError)
		end

		return
	end

	if (strData) then
		if (ticket.handler and type(ticket.handler) == 'function') then
			ticket.handler(strData, tHeaders)
		else
			dbg("[ReceivedAsync]: #"..responseCode.."#" .. strData .. "#")
		end
	end
end

function urlGet(url, headers, callback, errorHandler)
	local ticketId = C4:urlGet(url, headers)
	g_responseTickets[ticketId] = { handler = callback, errorHandler = errorHandler }
end

function urlPost(url, data, headers, callback, errorHandler)
	local ticketId = C4:urlPost(url, data, headers)
	g_responseTickets[ticketId] = { handler = callback, errorHandler = errorHandler }
end

function urlPut(url, data, headers, callback, errorHandler)
	local ticketId = C4:urlPut(url, data, headers)
	g_responseTickets[ticketId] = { handler = callback, errorHandler = errorHandler }
end

function urlEncode(str)
	return str:gsub(" ","+"):gsub("\n","\r\n"):gsub("([^%w])",function(ch)
			return string.format("%%%02X",string.byte(ch))
		end)
end

function pullUrlTicket(ticketId)
	local ticket = g_responseTickets[ticketId]
	g_responseTickets[ticketId] = nil
	return ticket
end

function toboolean(val)
	local rval = false;

	if type(val) == "string" and (string.lower(val) == 'true' or val == "1") then
		rval = true
	elseif type(val) == "number" and val ~= 0 then
		rval =  true
	elseif type(val) == "boolean" then
		rval = val
	end

	return rval
end

function tointeger(val)
	local nval = tonumber(val)
	return (nval >= 0) and math.floor(nval + 0.5) or math.ceil(nval - 0.5)
end

function tonumber_loc(str, base)
    local s = str:gsub(",", ".") -- Assume US Locale decimal separator
    local num = tonumber(s, base)

    if (num == nil) then
        s = str:gsub("%.", ",") -- Non-US Locale decimal separator
        num = tonumber(s, base)
    end

    return num
end

function IsEmpty(str)
	return str == nil or str == ""
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- API
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function ControllerInterface()
    local self = {}

    self.idBinding = 1
    self.prefix = "CTRL_"
    self.themes = {}
    self.effects = {}

    function self.GetLevel(channel)
    	local params = {}

    	params.CHANNEL = channel

    	SendToProxy(self.idBinding, self.prefix.."GET_LEVEL", params)
    end

    function self.SetLevel(channel, level, ramp_rate, start_delay)
    	local convertedLevel = ((level / 100) * 255)

    	local params = {}

    	params.CHANNEL = channel
    	params.LEVEL = convertedLevel

    	if (ramp_rate ~= nil) then
    		params.RAMP_RATE = ramp_rate
    	end

    	if (start_delay ~= nil) then
    		params.START_DELAY = start_delay
    	end

    	SendToProxy(self.idBinding, self.prefix.."SET_LEVEL", params)
        gCachedChannelLevels[tostring(channel)] = level
	end

    function self.Save(channel, slot)
        local params = {}

        params.CHANNEL = channel
        params.SAVE_SLOT = slot

        SendToProxy(self.idBinding, self.prefix.."SAVE", params)
    end

    function self.Recall(channel, slot)
        local params = {}

        params.CHANNEL = channel
        params.SAVE_SLOT = slot

        SendToProxy(self.idBinding, self.prefix.."RECALL", params)
    end

    function self.Animate(mode, channels, factors)
        local params = {}

        params.MODE = mode
        params.CHANNELS = channels
        params.FACTORS = factors

        SendToProxy(self.idBinding, self.prefix.."ANIMATE", params)
    end

    function self.AnimateColorCycle(saturation, intensity, ramp_rate)
        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2
        params.S = tonumber(saturation)
        params.V = tonumber(intensity)
        params.RAMP_RATE = tonumber(ramp_rate)

        SendToProxy(self.idBinding, self.prefix.."ANIMATE_COLOR_CYCLE", params)
    end

     function self.AnimateLevels(rate1, rate2, rate3, rate4)
        local params = {}

        channels = Translator.MapTo(LIGHT_PROXY_BINDING)
        for i=1, #channels do 
            params["CHANNEL_"..i] = channels[i]
        end

        params.RAMP_RATE_1 = tonumber(rate1)
        params.RAMP_RATE_2 = tonumber(rate2)
        params.RAMP_RATE_3 = tonumber(rate3)
        params.RAMP_RATE_4 = tonumber(rate4)

        SendToProxy(self.idBinding, self.prefix.."ANIMATE_LEVELS", params)
    end

    function self.GetThemes()
        SendToProxy(self.idBinding, self.prefix.."GET_THEMES", {})
    end

    function self.GetEffects()
        SendToProxy(self.idBinding, self.prefix.."GET_EFFECTS", {})
    end

    function self.SetTheme(theme)

        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2
        params.THEME = theme

        SendToProxy(self.idBinding, self.prefix.."SET_THEME", params)
    end

    function self.SetEffect(effect, interval, duration)

        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2
        params.EFFECT = effect
        params.INTERVAL = interval
        params.DURATION = duration

        SendToProxy(self.idBinding, self.prefix.."SET_EFFECT", params)
    end

    function self.StopEffect()

        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2

        SendToProxy(self.idBinding, self.prefix.."STOP_EFFECT", params)
    end

    return self
end

API = ControllerInterface()


--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Binding Translator
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function BindingToAPITranslator()
    local self = {}

    function self.MapTo(idBinding)
    	local mapping = {}

        local start = tonumber(gStartChannel)

    	if (idBinding == LIGHT_PROXY_BINDING) then
    		if (gChannelMode == "White") then
    			table.insert(mapping, start)
    		elseif(gChannelMode == "RGB") then
    			for i=0,2 do
    				table.insert(mapping, start + i)
    			end
    		elseif(gChannelMode == "RGBW") then
    			for i=0,3 do
    				table.insert(mapping, start + i)
    			end
    		end
		elseif(idBinding == RED_COLOR_SLIDER_BINDING) then
			table.insert(mapping, start)
		elseif(idBinding == GREEN_COLOR_SLIDER_BINDING) then
			table.insert(mapping, start + 1)
		elseif(idBinding == BLUE_COLOR_SLIDER_BINDING) then
			table.insert(mapping, start + 2)
		elseif(idBinding == WHITE_COLOR_SLIDER_BINDING) then
			table.insert(mapping, start + 3)
    	end

    	return mapping
    end

    function self.MapFrom(channel)
        local mapping = {}

        table.insert(mapping, LIGHT_PROXY_BINDING)

        local start = tonumber(gStartChannel)
        channel = tonumber(channel)

        if (start == channel) then
            table.insert(mapping, RED_COLOR_SLIDER_BINDING)
        elseif (start + 1 == channel) then
            table.insert(mapping, GREEN_COLOR_SLIDER_BINDING)
        elseif (start + 2 == channel) then
            table.insert(mapping, BLUE_COLOR_SLIDER_BINDING)
        elseif (start + 3 == channel) then
            table.insert(mapping, WHITE_COLOR_SLIDER_BINDING)
        end

        return mapping
    end

    return self
end

Translator = BindingToAPITranslator()


function ColorConvertor()
    local self = {}

    function self.rgbToHsv(r, g, b)
        r, g, b = r / 255, g / 255, b / 255
        local max, min = math.max(r, g, b), math.min(r, g, b)
        local h, s, v
        v = max

        local d = max - min
        if max == 0 then s = 0 else s = d / max end

        if max == min then
            h = 0 -- achromatic
        else
            if max == r then
                h = (g - b) / d
            if g < b then h = h + 6 end
            elseif max == g then h = (b - r) / d + 2
            elseif max == b then h = (r - g) / d + 4
            end
            h = h / 6
        end

        return h, s, v
    end

    function self.hsvToRgb(h, s, v)
        local r, g, b

        local i = math.floor(h * 6);
        local f = h * 6 - i;
        local p = v * (1 - s);
        local q = v * (1 - f * s);
        local t = v * (1 - (1 - f) * s);

        i = i % 6

        if i == 0 then r, g, b = v, t, p
        elseif i == 1 then r, g, b = q, v, p
        elseif i == 2 then r, g, b = p, v, t
        elseif i == 3 then r, g, b = p, q, v
        elseif i == 4 then r, g, b = t, p, v
        elseif i == 5 then r, g, b = v, p, q
        end

        return r * 255, g * 255, b * 255
    end

    function self.LevelToRGB(num)

        if (num < 0) then num = 0 end
        if (num > 100) then num = 100 end

        local step = 255 / 33

        local red = 0
        local green = 0
        local blue = 0

        if (num >= 0 and num < 33) then
            red = 255 - (step * num)
            green = step * num
        elseif (num >= 33 and num < 66) then
            num = num - 33
            green = 255 - (step * num)
            blue = step * num
        elseif (num < 100) then
            num = num - 66
            blue = 255 - (step * num)
            red = step * num
        else
            red = 255
            green = 255
            blue = 255
        end

        return math.floor(red), math.floor(green), math.floor(blue)
    end

    function self.RGBToLevel(red, green, blue)

        if (red < 0) then red = 0 end
        if (red > 255) then red = 255 end
        if (green < 0) then green = 0 end
        if (green > 255) then green = 255 end
        if (blue < 0) then blue = 0 end
        if (blue > 255) then blue = 255 end

        local num = 0
        local step = 255 / 33

        if (blue == 0) then
            num = green / step
        elseif (red == 0) then
            num = 33 + (blue / step)
        elseif (green == 0) then
            num = 66 + (red / step)
        else
            num = 100
        end

        return num
    end

    return self
end

Convertor = ColorConvertor()
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Main
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

if (PersistData == nil) then
    PersistData = {}
end

CHANNEL_MODES = {
	"White",
	"RGB",
	"RGBW"
}

gStartChannel		 = tostring(Properties["Start Channel"] - 1) or "No channel selected"
gChannelMode		 = Properties["Channel Mode"] or CHANNEL_MODES[1]
gDimmerLevel         = 0
gCachedChannelLevels = {}
gClickRampRateDown   = PersistData["ratedown"] or 50
gClickRampRateUp     = PersistData["rateup"] or 50
gPresetLevel         = PersistData["preset"] or 100

LIGHT_PROXY_BINDING 	   = 5001
DMX_CONTROLLER_BINDING	   = 1
RED_COLOR_SLIDER_BINDING   = 2
GREEN_COLOR_SLIDER_BINDING = 3
BLUE_COLOR_SLIDER_BINDING  = 4
WHITE_COLOR_SLIDER_BINDING = 5
RGB_COLOR_SLIDER_BINDING   = 6
ON_BINDING                 = 300
OFF_BINDING                = 301
TOGGLE_BINDING             = 302

function OnDriverInit()
    dbg("(OnDriverInit)")

    OnPropertyChanged("Channel Mode")
    OnPropertyChanged("Start Channel Selector")

    API.GetThemes()
 	API.GetEffects()
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Properties
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////


PROPS = {}

function PROPS.StartChannelSelector(channel)
	dbg("(StartChannelSelector) Channel("..channel..")")

	gStartChannel = tostring(channel - 1)

	C4:UpdateProperty("Start Channel", channel)

	-- Force light state change
	C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", {STATE=false})
	C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", {STATE=true})
end

function PROPS.ChannelModeSelector(mode)
	dbg("(ChannelModeSelector) Mode("..mode..")")

	gChannelMode = mode

	C4:UpdateProperty("Channel Mode", gChannelMode)
end

function COMMAND.SavePreset(tParams)
	dbg("(SavePreset) Slot("..tParams.Preset..")")

	-- 0 based index so minus 1
	local preset = tonumber(tParams.Preset) - 1

	channels = Translator.MapTo(LIGHT_PROXY_BINDING)
	for i=1, #channels do 
		API.Save(channels[i], preset)
	end
end

function COMMAND.RecallPreset(tParams)
	dbg("(RecallPreset) Slot("..tParams.Preset..")")

	-- 0 based index so minus 1
	local preset = tonumber(tParams.Preset) - 1

	channels = Translator.MapTo(LIGHT_PROXY_BINDING)
	for i=1, #channels do 
		API.Recall(channels[i], preset)
	end
end

function COMMAND.StartColorCycleAnimation(tParams)
	if (gChannelMode == "White") then return end

	dbg("(StartColorCycleAnimation) Starting Color Cycle Animation")

	local red, green, blue = tParams['Color Selector']:match("([^,]+),([^,]+),([^,]+)")

	PROXY_COMMANDS.RAMP_TO_LEVEL(RED_COLOR_SLIDER_BINDING, { LEVEL = ((red * 100) / 255) })
    PROXY_COMMANDS.RAMP_TO_LEVEL(GREEN_COLOR_SLIDER_BINDING, { LEVEL = ((green * 100) / 255) })
    PROXY_COMMANDS.RAMP_TO_LEVEL(BLUE_COLOR_SLIDER_BINDING, { LEVEL = ((blue * 100) / 255) })

    API.AnimateColorCycle(tParams.Saturation, tParams.Intensity, tParams["Ramp Rate"])
end

function COMMAND.StartLevelAnimation(tParams)
	dbg("(StartLevelAnimation) Starting Level Animation")

    API.AnimateLevels(tParams["Ramp Rate Channel 1"], tParams["Ramp Rate Channel 2"], tParams["Ramp Rate Channel 3"], tParams["Ramp Rate Channel 4"])
end

function COMMAND.StopAnimation(tParams)
	dbg("(StopAnimation) Stopping current animation")

	API.Animate(-1)
end

function COMMAND.SetTheme(tParams)
	dbg("(SetTheme) Setting Theme "..tParams.Theme)

	API.SetTheme(tParams.Theme)
end

function COMMAND.SetEffect(tParams)
	dbg("(SetEffect) Setting Effect "..tParams.Effect)

	local red, green, blue = tParams['Color Selector']:match("([^,]+),([^,]+),([^,]+)")

	PROXY_COMMANDS.RAMP_TO_LEVEL(RED_COLOR_SLIDER_BINDING, { LEVEL = ((red * 100) / 255) })
    PROXY_COMMANDS.RAMP_TO_LEVEL(GREEN_COLOR_SLIDER_BINDING, { LEVEL = ((green * 100) / 255) })
    PROXY_COMMANDS.RAMP_TO_LEVEL(BLUE_COLOR_SLIDER_BINDING, { LEVEL = ((blue * 100) / 255) })

	API.SetEffect(tParams.Effect, tParams.Interval, tParams.Duration)
end

function COMMAND.StopEffect(tParams)
	dbg("(StopEffect) Stopping current effect")

	API.StopEffect()
end

function GetThemesForProgramming(currentValue, done, search, searchFilter)
	dbg('GetThemesForProgramming')
	local list = {}

	for k, v in pairs(API.themes) do
		table.insert (list, { text = tostring(v), value = tostring(v) })
	end

	return (list)
end

function GetEffectsForProgramming(currentValue, done, search, searchFilter)
	dbg('GetEffectsForProgramming')
	local list = {}

	for k, v in pairs(API.effects) do
		table.insert (list, { text = tostring(v), value = tostring(v) })
	end

	return (list)
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Proxy
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

PROXY_COMMANDS = {}

function ReceivedFromProxy(idBinding, strCommand, tParams)
	dbg("(ReceivedFromProxy) idBinding(" .. idBinding .. ") strCommand(" .. strCommand .. ")")
	tParams = tParams or {}

	if (gChannel == "No channel selected") then
    dbg("Please select a channel via the channel selector property")
    return
	end

	if (PROXY_COMMANDS[strCommand] ~= nil and type(PROXY_COMMANDS[strCommand]) == 'function') then
		  PROXY_COMMANDS[strCommand](idBinding, tParams)
    else
      dbg('No Proxy Function found')
	end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- From Light Proxy
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

BUTTON_IDS = {
    "TOP",
    "BOTTOM",
    "TOGGLE"
}

BUTTON_ACTIONS = {
    "RELEASE",
    "PRESS",
    "SINGLE CLICK",
    "DOUBLE CLICK",
    "TRIPLE CLICK"
}

function PROXY_COMMANDS.GET_CONNECTED_STATE(idBinding, tParams)
	dbg("(GET_CONNECTED_STATE)")

	C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", { STATE = true })
end

function PROXY_COMMANDS.GET_LIGHT_LEVEL(idBinding, tParams)
  dbg("(GET_LIGHT_LEVEL)")

  channels = Translator.MapTo(idBinding)
  for i=1, #channels do 
    API.GetLevel(channels[i])
  end
end

function PROXY_COMMANDS.SET_LEVEL(idBinding, tParams)
	dbg("(SET_LEVEL) Level("..tostring(tParams.LEVEL)..")")

  PROXY_COMMANDS.RAMP_TO_LEVEL(idBinding, tParams)
end

function PROXY_COMMANDS.RAMP_TO_LEVEL(idBinding, tParams)
  if (gDimmerLevel > tonumber(tParams.LEVEL)) then
    tParams.TIME = gClickRampRateDown
  else
    tParams.TIME = gClickRampRateUp
  end

  dbg("(RAMP_TO_LEVEL) Level("..tostring(tParams.LEVEL)..") Time("..tParams.TIME..")")

  channels = Translator.MapTo(idBinding)
  for i=1, #channels do 
    API.SetLevel(channels[i], tParams.LEVEL, tParams.TIME)
  end
end

function PROXY_COMMANDS.SET_PRESET_LEVEL(idBinding, tParams)
	gPresetLevel = tParams["LEVEL"]
	PersistData["preset"] = gPresetLevel

	dbg("(SET_PRESET_LEVEL) Level("..gPresetLevel..")")
end

function PROXY_COMMANDS.SET_CLICK_RATE_DOWN(idBinding, tParams)
  local temp = tonumber(tParams.RATE)

  if (temp >= 1000) then
    temp = math.floor((100/(temp/1000))) -- convert to seconds
  else
    temp = 0
  end

  gClickRampRateDown = temp

	PersistData["ratedown"] = gClickRampRateDown

  dbg("(SET_CLICK_RATE_DOWN) Raw("..tParams.RATE..") Rate(" .. gClickRampRateDown .. "%)")
end

function PROXY_COMMANDS.SET_CLICK_RATE_UP(idBinding, tParams)
  local temp = tonumber(tParams.RATE)

  if (temp >= 1000) then
    temp = math.floor((100/(temp/1000))) -- convert to seconds
  else
    temp = 0
  end

  gClickRampRateUp = temp
	
	PersistData["rateup"] = gClickRampRateUp

  dbg("(SET_CLICK_RATE_UP) Raw("..tParams.RATE..") Rate(" .. gClickRampRateUp .. "%)")
end

function PROXY_COMMANDS.BUTTON_ACTION(idBinding, tParams)
	local btn = BUTTON_IDS[tonumber(tParams["BUTTON_ID"])+1]
	local act = BUTTON_ACTIONS[tonumber(tParams["ACTION"]+1)]

	dbg("(BUTTON_ACTION) BUTTON(".. btn ..") ACTION(" .. act ..")")

	if (btn == "TOGGLE") then
		if (act == "SINGLE CLICK") then
			PROXY_COMMANDS.TOGGLE(idBinding, tParams)
		end
	end
end

function PROXY_COMMANDS.ON(idBinding, tParams)
	dbg("(ON)")

	channels = Translator.MapTo(idBinding)
	for i=1, #channels do 
		API.SetLevel(channels[i], gPresetLevel)
	end
end

function PROXY_COMMANDS.OFF(idBinding, tParams)
	dbg("(OFF)")

	channels = Translator.MapTo(idBinding)
	for i=1, #channels do 
		API.SetLevel(channels[i], 0)
	end
end

function PROXY_COMMANDS.TOGGLE(idBinding, tParams)
  dbg("(TOGGLE)")

  if (gDimmerLevel == 0) then
    value =  gPresetLevel
  else
    value = 0
  end 

  channels = Translator.MapTo(idBinding)
  for i=1, #channels do 
    API.SetLevel(channels[i], value)
  end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- From DMX Controller 
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////


function PROXY_COMMANDS.CTRL_GET_LEVEL(idBinding, tParams)
	dbg("(CTRL_GET_LEVEL) Channel("..tostring(tParams.CHANNEL)..") Level("..tostring(tParams.LEVEL)..")")

  if (gStartChannel == tParams.CHANNEL) then
    gDimmerLevel = tonumber(tParams.LEVEL)
  end

  bindings = Translator.MapFrom(tParams.CHANNEL)

  for i=1, #bindings do 
    UpdateProxy(bindings[i], tParams.LEVEL)
  end
end

function PROXY_COMMANDS.CTRL_SET_LEVEL(idBinding, tParams)
	dbg("(CTRL_SET_LEVEL) Channel("..tostring(tParams.CHANNEL)..")")

  if (gStartChannel == tParams.CHANNEL) then
    gDimmerLevel = tonumber(gCachedChannelLevels[tostring(tParams.CHANNEL)]) or 0
  end

  bindings = Translator.MapFrom(tParams.CHANNEL)

  for i=1, #bindings do 
    UpdateProxy(bindings[i], gCachedChannelLevels[tostring(tParams.CHANNEL)])
  end
end

function PROXY_COMMANDS.CTRL_GET_THEMES(idBinding, tParams)
  dbg("(CTRL_GET_THEMES)")

  API.themes = {}

  for k,v in pairs(tParams) do
    table.insert(API.themes, v)
  end
end

function PROXY_COMMANDS.CTRL_GET_EFFECTS(idBinding, tParams)
  dbg("(CTRL_GET_EFFECTS)")

  API.effects = {}

  for k,v in pairs(tParams) do
    table.insert(API.effects, v)
  end
end


--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- From Sliders 
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function PROXY_COMMANDS.SET_COLOR_LEVEL(idBinding, tParams)
  dbg("(SET_COLOR_LEVEL) Binding("..idBinding..") Level("..tParams.LEVEL..")")

  -- Do something special for RGB Color Slider
  if (idBinding == RGB_COLOR_SLIDER_BINDING) then
    local red, green, blue = Convertor.LevelToRGB(tonumber(tParams.LEVEL))
    PROXY_COMMANDS.RAMP_TO_LEVEL(RED_COLOR_SLIDER_BINDING, { LEVEL = ((red * 100) / 255) })
    PROXY_COMMANDS.RAMP_TO_LEVEL(GREEN_COLOR_SLIDER_BINDING, { LEVEL = ((green * 100) / 255) })
    PROXY_COMMANDS.RAMP_TO_LEVEL(BLUE_COLOR_SLIDER_BINDING, { LEVEL = ((blue * 100) / 255) })
  else
    PROXY_COMMANDS.RAMP_TO_LEVEL(idBinding, tParams)
  end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Update
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function UpdateProxy(idBinding, value)
	dbg("(UpdateProxy) idBinding("..idBinding..") Level("..(value or "nil")..")")
	
	C4:SendToProxy(idBinding, "LIGHT_LEVEL", tostring(value))
  C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", { STATE = true })

  if (idBinding == LIGHT_PROXY_BINDING) then
    SyncLEDS()
  end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Binding Changed
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function OnBindingChanged(idBinding, strClass, bIsBound)
    if(bIsBound) then
        dbg("(OnBindingChanged) idBinding("..idBinding..") Class("..strClass..") Bound(True)")

        if (idBinding == RGB_COLOR_SLIDER_BINDING) then return end

        PROXY_COMMANDS.GET_LIGHT_LEVEL(idBinding, tParams)

        if (idBinding == DMX_CONTROLLER_BINDING) then
          API.GetThemes()
          API.GetEffects()
        end
    end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Proxy Send
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function SendToProxy(idBinding, strCommand, tParams)
  dbg("(SendToProxy) idBinding("..idBinding..") Command("..strCommand..")")

  C4:SendToProxy(idBinding, strCommand, tParams)
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- ADVANCED LIGHTING SCENES
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

sceneCollection     = PersistData["sceneCollection"] or {}
flashCollection     = PersistData["flashCollection"] or {}
elementCounter      = 0
currentScene        = 0
executeElementTimer = 0

function PROXY_COMMANDS.PUSH_SCENE(idBinding, tParams)
    for k,v in pairs(tParams) do
        dbg(k .. ": " .. v)
    end
    local sceneNum = tParams["SCENE_ID"]
    local elements = tParams["ELEMENTS"]
    local flash = tParams["FLASH"]
    dbg("scene_id: " .. sceneNum)
    dbg("elements: " .. tParams["ELEMENTS"])
    dbg("flash: " .. tParams["FLASH"])
    local elementTable = collect(elements)
    local scene = {}
    for i=1,#elementTable do
        local t = {}
        t["Delay"] = elementTable[i][1][1]
        t["Rate"] = elementTable[i][2][1]
        t["Level"] = elementTable[i][3][1]
        table.insert(scene,t)
    end
    sceneCollection[sceneNum] = scene
    flashCollection[sceneNum] = flash
    PersistData["sceneCollection"] = sceneCollection
    PersistData["flashCollection"] = flashCollection
end

function PROXY_COMMANDS.REMOVE_SCENE(idBinding, tParams)
    local sceneNum = tParams["SCENE_ID"]
    dbg("scene_id: " .. sceneNum)
    sceneCollection[sceneNum] = nil
    flashCollection[sceneNum] = nil
    PersistData["sceneCollection"] = sceneCollection
    PersistData["flashCollection"] = flashCollection
end

function PROXY_COMMANDS.ACTIVATE_SCENE(idBinding, tParams)
    local sceneNum = tParams["SCENE_ID"]
    for k,v in pairs(tParams) do
        dbg(k .. ": " .. v)
    end
    currentScene = sceneNum
    elementCounter = 0
    playScene()
end

function playScene()
    dbg("playScene")
    if (elementCounter ~= 0) then
        local t = {}
        t["LEVEL"] = sceneCollection[tostring(currentScene)][elementCounter]["Level"] -- Given range 0-100
        local temp =  math.floor((100/(sceneCollection[tostring(currentScene)][elementCounter]["Rate"]/1000))) -- Given in milliseconds from either ms, sec, or min. (2ms, 2000ms(sec),120000ms(min))
                                                                                                               -- convert to seconds
        if(temp >= 0  and temp <= 100) then
            t["TIME"] = temp
        else
            t["TIME"] = 100
        end

        PROXY_COMMANDS.RAMP_TO_LEVEL(LIGHT_PROXY_BINDING, t)
    end
    elementCounter = elementCounter + 1
    dbg("elementCounter is now " .. elementCounter)

    if (elementCounter > #sceneCollection[tostring(currentScene)] and flashCollection[tostring(currentScene)] == "1") then
        elementCounter = 1
    end

    if (elementCounter <= #sceneCollection[tostring(currentScene)]) then
        local timeInterval = sceneCollection[tostring(currentScene)][elementCounter]["Delay"] or -1
        executeElementTimer = killTimer(executeElementTimer)
        executeElementTimer = startTimer(timeInterval, "MILLISECONDS", playScene )
    else
        dbg("end of scene")
    end

end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- ADVANCED LIGHTING SCENES HELPER FUNCTIONS
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function parseargs(s)
  local arg = {}
  string.gsub(s, "(%w+)=([\"'])(.-)%2", function (w, _, a)
    arg[w] = a
  end)
  return arg
end


function collect(s)
  local stack = {}
  local top = {}
  table.insert(stack, top)
  local ni,c,label,xarg, empty
  local i, j = 1, 1
  while true do
    ni,j,c,label,xarg, empty = string.find(s, "<(%/?)([%w:]+)(.-)(%/?)>", i)
    if not ni then break end
    local text = string.sub(s, i, ni-1)
    if not string.find(text, "^%s*$") then
      table.insert(top, text)
    end
    if empty == "/" then  -- empty element tag
      table.insert(top, {label=label, xarg=parseargs(xarg), empty=1})
    elseif c == "" then   -- start tag
      top = {label=label, xarg=parseargs(xarg)}
      table.insert(stack, top)   -- new level
    else  -- end tag
      local toclose = table.remove(stack)  -- remove top
      top = stack[#stack]
      if #stack < 1 then
        error("nothing to close with "..label)
      end
      if toclose.label ~= label then
        error("trying to close "..toclose.label.." with "..label)
      end
      table.insert(top, toclose)
    end
    i = j+1
  end
  local text = string.sub(s, i)
  if not string.find(text, "^%s*$") then
    table.insert(stack[#stack], text)
  end
  if #stack > 1 then
    error("unclosed "..stack[#stack].label)
  end
  return stack[1]
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Keypad Functions
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

BUTTON_LINKS = {}

function BUTTON_LINKS.DO_PUSH(id, tParams)
	dbg("(DO_PUSH) id:(" .. id .. ")")
	
	if (id == ON_BINDING) then
        PROXY_COMMANDS.ON(LIGHT_PROXY_BINDING, tParams)
    elseif (id == OFF_BINDING) then
        PROXY_COMMANDS.OFF(LIGHT_PROXY_BINDING, tParams)
    elseif (id == TOGGLE_BINDING) then
        PROXY_COMMANDS.TOGGLE(LIGHT_PROXY_BINDING, tParams)
    end
end

function BUTTON_LINKS.DO_RELEASE(id, tParams)
	dbg("(DO_RELEASE) id:(" .. id .. ")")
end

function BUTTON_LINKS.DO_CLICK(id, tParams)
	dbg("(DO_CLICK) id:(" .. id .. ")")

    if (id == ON_BINDING) then
        PROXY_COMMANDS.ON(LIGHT_PROXY_BINDING, tParams)
    elseif (id == OFF_BINDING) then
        PROXY_COMMANDS.OFF(LIGHT_PROXY_BINDING, tParams)
    elseif (id == TOGGLE_BINDING) then
        PROXY_COMMANDS.TOGGLE(LIGHT_PROXY_BINDING, tParams)
    end
end

function SyncLEDS()
	dbg("(SyncLEDS) LEVEL(".. gDimmerLevel ..")")

	local led = tonumber(gDimmerLevel) > 0 and true or false
	
    C4:SendToProxy(OFF_BINDING, "MATCH_LED_STATE", {STATE=not led})
    C4:SendToProxy(ON_BINDING, "MATCH_LED_STATE", {STATE=led})
    C4:SendToProxy(TOGGLE_BINDING, "MATCH_LED_STATE", {STATE=led})
end

