--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Command Handler
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

COMMAND = {}

function ExecuteCommand(strCommand, tParams)
	trace('ExecuteCommand', strCommand)
	tParams = tParams or {}
	local strCmd = string.gsub(strCommand, " ","")
	if (strCommand == 'LUA_ACTION') then
		for cmd,cmdv in pairs(tParams) do
			if (cmd == 'ACTION') then
				if (COMMAND[cmdv] ~= nil and type(COMMAND[cmdv]) == 'function') then
					COMMAND[cmdv](tParams)
				else
					dbg('Unknown Action Command.')
				end
			end
		end
	else
		if (COMMAND[strCmd] ~= nil and type(COMMAND[strCmd]) == 'function') then
			COMMAND[strCmd](tParams)
		else
			dbg('Unknown command.')
		end
	end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Debugging
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function dbg(...)
	if (Properties['DEBUG MODE'] == 'ON') then print(...) end
end

function trace(strFn, ...)
	if (Properties['DEBUG MODE'] ~= 'ON') then return end
	for k,v in pairs(arg) do arg[k] = tostring(v) end
	dbg(string.format('[%s](%s)', strFn, table.concat(arg, ', ')))
end

function err(...)
	if (Properties['DEBUG MODE'] ~= 'ON') then return end
	for k,v in pairs(arg) do arg[k] = tostring(v) end
	dbg(string.format('[ERROR]: %s', table.concat(arg, ', ')))
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--HOUSELOGIX EXCEPTION HANDLING ROUTINES
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function NilException(...)
    local throw = false
    local msg = ""
    for i=1,select("#",...) do
        local temp = select(i,...)
        if (temp) then
            if (i == 1) then
                temp = string.format("(%s): ", temp)
            elseif (type(temp) == "string") then
                temp = string.format("\"%s\"",temp)
            end
            msg = msg .. temp .. " "
        else
            msg = msg .. "NIL PARAM"
            throw = true
        end
    end
    if (throw) then msg = msg .. " -- THROWING AWAY" end
    dbg(msg)
    return throw
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Properties handler
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

PROPS = {}

function OnPropertyChanged(strProperty)
	trace('OnPropertyChanged', strProperty)
	local strProp = string.gsub(strProperty, ' ','')
	if (PROPS[strProp] ~= nil and type(PROPS[strProp]) == 'function') then
		PROPS[strProp](Properties[strProperty])
	end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Lua table utils
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function PrintTable(tValue, sIndent)
	sIndent = sIndent or '   '
	for k,v in pairs(tValue) do
		print(sIndent .. tostring(k) .. ':  ' .. tostring(v))
		if (type(v) == 'table') then
			PrintTable(v, sIndent .. '   ')
		end
	end
end

function AlphabeticalPairs(t, f)
	local a = {}

	for n in pairs(t) do
		table.insert(a, n)
	end

	table.sort(a, f)

	local index = 0      -- iterator variable
	local iter = function ()   -- iterator function
			index = index + 1
			if a[index] == nil then
				return nil
			else
				return a[index], t[a[index]]
			end
		end

	return iter
end

-- Be aware, this is really a hack. You could get the wrong key for a value,
-- especially if your data has duplicate values.
function AlphabeticalPairsByValue(t, f)
	local a = {}

	for k,v in pairs(t) do
		table.insert(a, v)
	end

	table.sort(a, f)

	local index = 0
	local iter = function()
			index = index + 1
			if a[index] == nil then
				return nil
			else
				for k,v in pairs(t) do
					if v == a[index] then
						return k, v
					end
				end
			end
		end

	return iter
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Timer Helpers
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

if not HOUSELOGIX then HOUSELOGIX = {} end

function OnTimerExpired(idTimer)
	local timer = pullTimer(idTimer)
	if (not timer) then dbg("UNKNOWN TIMER FIRED | THROWING AWAY") return end
	timer.handler()
end

--------------------------------
-- Helpers
--------------------------------
g_timers = {}

function startTimer(value, units, callback)
	local timerId = C4:AddTimer(tonumber(value), units)
	g_timers[timerId] = { handler = callback }
	return timerId
end

function pullTimer(timerId)
	local timer = g_timers[timerId]
	g_timers[timerId] = nil
	return timer
end

function killTimer(timerId)
	local timer = g_timers[timerId]
	g_timers[timerId] = nil
	if tonumber(timerId) > 0 then
		C4:KillTimer(timerId)
	end
	return timer
end

--------------------------------
-- Classes
--------------------------------

--------------------------------
-- Resets self if started again before expiration
--------------------------------
function HOUSELOGIX.SingletonTimer(nInterval, strUnits, fnCallback)
	local self = {}

	local ID = 0
	local interval = nInterval
	local units = strUnits
	local callback = fnCallback

	function self.Start()
		if ID ~= 0 then
			killTimer(ID)
		end

		ID = startTimer(interval, units, callback)
	end

	function self.Stop()
		if ID ~= 0 then
			killTimer(ID)
			ID = 0
		end
	end

	return self
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- URL Callbacks
-- HOUSELOGIX 2015
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

g_responseTickets = {}

function ReceivedAsync(ticketId, strData, responseCode, tHeaders, strError)
	local ticket = pullUrlTicket(ticketId)
	if (not ticket) then dbg("[ReceivedAsync]: Unknown Ticket | Throwing Away") return end

	if (strError) then
		if (ticket.errorHandler and type(ticket.errorHandler) == 'function') then
			ticket.errorHandler(strError)
		else
			dbg("[ReceivedAsync]: ERROR | " .. strError)
		end

		return
	end

	if (strData) then
		if (ticket.handler and type(ticket.handler) == 'function') then
			ticket.handler(strData, tHeaders)
		else
			dbg("[ReceivedAsync]: #"..responseCode.."#" .. strData .. "#")
		end
	end
end

function urlGet(url, headers, callback, errorHandler)
	local ticketId = C4:urlGet(url, headers)
	g_responseTickets[ticketId] = { handler = callback, errorHandler = errorHandler }
end

function urlPost(url, data, headers, callback, errorHandler)
	local ticketId = C4:urlPost(url, data, headers)
	g_responseTickets[ticketId] = { handler = callback, errorHandler = errorHandler }
end

function urlPut(url, data, headers, callback, errorHandler)
	local ticketId = C4:urlPut(url, data, headers)
	g_responseTickets[ticketId] = { handler = callback, errorHandler = errorHandler }
end

function urlEncode(str)
	return str:gsub(" ","+"):gsub("\n","\r\n"):gsub("([^%w])",function(ch)
			return string.format("%%%02X",string.byte(ch))
		end)
end

function pullUrlTicket(ticketId)
	local ticket = g_responseTickets[ticketId]
	g_responseTickets[ticketId] = nil
	return ticket
end

function toboolean(val)
	local rval = false;

	if type(val) == "string" and (string.lower(val) == 'true' or val == "1") then
		rval = true
	elseif type(val) == "number" and val ~= 0 then
		rval =  true
	elseif type(val) == "boolean" then
		rval = val
	end

	return rval
end

function tointeger(val)
	local nval = tonumber(val)
	return (nval >= 0) and math.floor(nval + 0.5) or math.ceil(nval - 0.5)
end

function tonumber_loc(str, base)
    local s = str:gsub(",", ".") -- Assume US Locale decimal separator
    local num = tonumber(s, base)

    if (num == nil) then
        s = str:gsub("%.", ",") -- Non-US Locale decimal separator
        num = tonumber(s, base)
    end

    return num
end

function IsEmpty(str)
	return str == nil or str == ""
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Advanced Lighting Scenes Handling
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function AdvancedLightingScenes()
    local self = {}

    self.Scenes = {}
    self.Timers = {}

    function self.Push(idBinding, tParams)
        self.dbg("[Push] - " .. self.StringtParams(tParams))

        self.Scenes[tointeger(tParams.SCENE_ID)] = {
            elements = self.Elements2Table(tParams.ELEMENTS),
            flash = toboolean(tParams.FLASH),
            ignoreRamp = toboolean(tParams.IGNORE_RAMP),
            fromGroup = toboolean(tParams.FROM_GROUP)
        }

        self.Save()

        self.dbg("[Push] - Saved Scene (" .. tostring(tParams.SCENE_ID) .. ") Actions(" .. #self.Scenes[tointeger(tParams.SCENE_ID)].elements .. ")")
    end

    function self.Remove(idBinding, tParams)
        self.dbg("[Remove] - " .. self.StringtParams(tParams))

        self.Scenes[tointeger(tParams.SCENE_ID)] = nil

        self.Save()

        self.dbg("[Remove] - Removed Scene (" .. tostring(tParams.SCENE_ID) .. ")")
    end

    function self.Activate(idBinding, tParams)
        self.dbg("[Activate] - " .. self.StringtParams(tParams))

        self.StopAll()

        local sceneId = tointeger(tParams.SCENE_ID)
        if (self.Scenes[sceneId] == nil) then
            self.err("[Activate] - No scene exists for (" .. tostring(sceneId) .. ")")
            return
        end

        self.Scenes[sceneId].state = 0
        self.Do(tointeger(tParams.SCENE_ID))
    end

    function self.StopAll()
        for sceneId, _ in pairs(self.Timers) do
            self.Timers[sceneId]:Cancel()
            self.Timers[sceneId] = nil
        end
    end

    function self.Do(sceneId)
        self.Timers[sceneId] = nil

        while (self.Scenes[sceneId] ~= nil) do
            local action = math.floor(self.Scenes[sceneId].state / 2) + 1
            if (action > #self.Scenes[sceneId].elements) then
                -- We've reached the end of the elements
                if (self.Scenes[sceneId].flash) then
                    -- It's set to flash, so start it all over again
                    self.Scenes[sceneId].state = 0
                    self.dbg("[Do] - Restarting Scene (" .. tostring(sceneId) .. ")")
                else
                    -- We're all done!
                    self.dbg("[Do] - Finished Scene (" .. tostring(sceneId) .. ")")
                    return
                end
            else
                local currentAction = self.Scenes[sceneId].elements[action]
                local sleep = currentAction.delay

                if (self.Scenes[sceneId].state % 2 == 1) then
                    self.dbg("[Do] - Executing Scene (" .. tostring(sceneId) .. ") Action (" .. tostring(action) .. ")")

                    local brightnessRate = currentAction.rate or 0
                    local colorRate = currentAction.colorRate or 0

                    if (currentAction.levelEnabled == true and currentAction.colorEnabled == true) then
                        -- Both brightness and color at the same time
                        -- TODO: Have a command for this!

                        -- Set the brightness so that calling color sets the color to this brightness
                        local s = {
                            LIGHT_BRIGHTNESS_TARGET = currentAction.level,
                            RATE = brightnessRate
                        }
                        RGB.SetBrightnessTarget(s)

                        startTimer(colorRate, "MILLISECONDS", function()
                            UpdateProxyLightBrightnessChanged(currentAction.level)
                        end)

                        local t = {
                            LIGHT_COLOR_TARGET_X = currentAction.colorX,
                            LIGHT_COLOR_TARGET_Y = currentAction.colorY,
                            RATE = colorRate,
                            LIGHT_COLOR_TARGET_MODE = currentAction.colorMode
                        }

                        PROXY_COMMANDS.SET_COLOR_TARGET(1, t)

                        sleep = (brightnessRate > colorRate) and brightnessRate or colorRate
                    elseif (currentAction.levelEnabled == true) then
                        -- Brightness only
                        -- TODO: The function for setting brightness

                        local t = {
                            LIGHT_BRIGHTNESS_TARGET = currentAction.level,
                            RATE = brightnessRate
                        }

                        PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(1, t)

                        sleep = brightnessRate
                    elseif (currentAction.colorEnabled == true) then
                        -- Color only
                        -- TODO: The function for setting color

                        local t = {
                            LIGHT_COLOR_TARGET_X = currentAction.colorX,
                            LIGHT_COLOR_TARGET_Y = currentAction.colorY,
                            RATE = colorRate,
                            LIGHT_COLOR_TARGET_MODE = currentAction.colorMode
                        }

                        PROXY_COMMANDS.SET_COLOR_TARGET(1, t)

                        sleep = colorRate
                    end
                end

                self.Scenes[sceneId].state = self.Scenes[sceneId].state + 1

                if (sleep > 0) then
                    -- Sleep until the end of this action to perform the next element
                    self.Timers[sceneId] = C4:SetTimer(sleep, function(oTimer) self.Do(sceneId) end)
                    self.dbg("[Do] - Sleeping to next Scene Action, Scene (" .. tostring(sceneId) .. ") Sleep (" .. tostring(sleep) .. ")")
                    return
                end
            end
        end
    end

    function self.Load()
        self.Scenes = PersistData['AdvancedLightingScenes'] or {}
    end

    function self.Save()
        PersistData['AdvancedLightingScenes'] = self.Scenes
    end

    function self.Elements2Table(elements)
        if (elements == nil) then return {} end

        local xml = C4:ParseXml('<d>' .. elements .. '</d>')
        if (xml == nil) then return {} end

        local result = {}

        for _, elementXml in ipairs(xml.ChildNodes) do
            if (elementXml.Name == 'element') then
                local t = {
                    delay = nil,
                    rate = 0,
                    colorRate = 0,
                    colorX = nil,
                    colorY = nil,
                    colorMode = nil,
                    level = nil,
                    levelEnabled = false,
                    colorEnabled = false
                }

                for _, dataXml in ipairs(elementXml.ChildNodes) do
                    if (dataXml.Name == 'delay') then
                        t.delay = tointeger(dataXml.Value)
                    elseif (dataXml.Name == 'rate' or dataXml.Name == 'brightnessRate') then
                        t.rate = tointeger(dataXml.Value)
                    elseif (dataXml.Name == 'level' or dataXml.Name == 'brightness') then
                        t.level = tointeger(dataXml.Value)
                    elseif (dataXml.Name == 'brightnessEnabled') then
                        t.levelEnabled = toboolean(dataXml.Value)
                    elseif (dataXml.Name == 'colorEnabled') then
                        t.colorEnabled = toboolean(dataXml.Value)
                    elseif (dataXml.Name == 'colorRate') then
                        t.colorRate = tointeger(dataXml.Value)
                    elseif (dataXml.Name == 'colorX') then
                        t.colorX = tonumber_loc(dataXml.Value)
                    elseif (dataXml.Name == 'colorY') then
                        t.colorY = tonumber_loc(dataXml.Value)
                    elseif (dataXml.Name == 'colorMode') then
                        t.colorMode = tointeger(dataXml.Value)
                    end
                end

                if (t.delay ~= nil) then
                    table.insert(result, t)
                end
            end
        end

        return result
    end

    function self.dbg(msg)
        dbg("[ALS] {DEBUG}: " .. msg)
    end

    function self.err(msg)
        print("[ALS] {ERROR}: " .. msg)
    end

    function self.StringtParams(tParams)
        if (tParams == nil) then return "" end

        local ret = ""

        for k, v in pairs(tParams) do
            ret = ret .. tostring(k) .. "(" .. tostring(v) .. ") "
        end

        return ret
    end

    return self
end

ALS = AdvancedLightingScenes()
ALS.Load()

PROXY_COMMANDS = PROXY_COMMANDS or {}

function PROXY_COMMANDS.PUSH_SCENE(idBinding, tParams)
    return ALS.Push(idBinding, tParams)
end

function PROXY_COMMANDS.REMOVE_SCENE(idBinding, tParams)
    return ALS.Remove(idBinding, tParams)
end

function PROXY_COMMANDS.ACTIVATE_SCENE(idBinding, tParams)
    return ALS.Activate(idBinding, tParams)
end

function PROXY_COMMANDS.RAMP_SCENE_UP(idBinding, tParams)
    dbg("[RAMP_SCENE_UP] idBinding(" .. idBinding .. ") " .. ALS.StringtParams(tParams))
end

function PROXY_COMMANDS.RAMP_SCENE_DOWN(idBinding, tParams)
    dbg("[RAMP_SCENE_DOWN] idBinding(" .. idBinding .. ") " .. ALS.StringtParams(tParams))
end

function PROXY_COMMANDS.STOP_SCENE_RAMP(idBinding, tParams)
    dbg("[STOP_SCENE_RAMP] idBinding(" .. idBinding .. ") " .. ALS.StringtParams(tParams))
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- API
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function ControllerInterface()
    local self = {}

    self.idBinding = 1
    self.prefix = "CTRL_"
    self.themes = {}
    self.effects = {}
    self.adv_effects = {}

    function self.GetLevel(channel)
    	local params = {}

    	params.CHANNEL = channel

    	SendToProxy(self.idBinding, self.prefix.."GET_LEVEL", params)
    end

    function self.SetLevel(channel, level, ramp_rate, start_delay)
    	local params = {}

    	params.CHANNEL = channel
    	params.LEVEL = level

    	if (ramp_rate ~= nil) then
    		params.RAMP_RATE = ramp_rate
    	end

    	if (start_delay ~= nil) then
    		params.START_DELAY = start_delay
    	end

    	SendToProxy(self.idBinding, self.prefix.."SET_LEVEL", params)
	end
	
    function self.Stop(channel)
    	local params = {}

    	params.CHANNEL = channel

    	SendToProxy(self.idBinding, self.prefix.."STOP", params)
    end

    function self.Save(channel, slot)
        local params = {}

        params.CHANNEL = channel
        params.SAVE_SLOT = slot

        SendToProxy(self.idBinding, self.prefix.."SAVE", params)
    end

    function self.Recall(channel, slot)
        local params = {}

        params.CHANNEL = channel
        params.SAVE_SLOT = slot

        SendToProxy(self.idBinding, self.prefix.."RECALL", params)
    end

    function self.Animate(mode, channels, factors)
        local params = {}

        params.MODE = mode
        params.CHANNELS = channels
        params.FACTORS = factors

        SendToProxy(self.idBinding, self.prefix.."ANIMATE", params)
    end

    function self.AnimateColorCycle(saturation, intensity, ramp_rate)
        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2
        params.S = tonumber(saturation)
        params.V = tonumber(intensity)
        params.RAMP_RATE = tonumber(ramp_rate)

        SendToProxy(self.idBinding, self.prefix.."ANIMATE_COLOR_CYCLE", params)
    end

     function self.AnimateLevels(rate1, rate2, rate3, rate4)
        local params = {}

        channels = Translator.MapTo(LIGHT_PROXY_BINDING)
        for i=1, #channels do
            params["CHANNEL_"..i] = channels[i]
        end

        params.RAMP_RATE_1 = tonumber(rate1)
        params.RAMP_RATE_2 = tonumber(rate2)
        params.RAMP_RATE_3 = tonumber(rate3)
        params.RAMP_RATE_4 = tonumber(rate4)

        SendToProxy(self.idBinding, self.prefix.."ANIMATE_LEVELS", params)
    end

    function self.GetThemes()
        SendToProxy(self.idBinding, self.prefix.."GET_THEMES", {})
    end

    function self.GetEffects()
        SendToProxy(self.idBinding, self.prefix.."GET_EFFECTS", {})
    end

    function self.GetAdvEffects()
        SendToProxy(self.idBinding, self.prefix.."GET_ADVANCED_EFFECTS", {})
    end

    function self.SetTheme(theme)

        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2
        params.THEME = theme

        SendToProxy(self.idBinding, self.prefix.."SET_THEME", params)
    end

    function self.SetEffect(effect, interval, duration)

        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2

        if (is_rgbw() or is_rgbww()) then
            params.CHANNEL_W = gStartChannel + 3
        end

        if (is_rgbww()) then
            params.CHANNEL_WW = gStartChannel + 4
        end

        params.EFFECT = effect
        params.INTERVAL = interval
        params.DURATION = duration

        SendToProxy(self.idBinding, self.prefix.."SET_EFFECT", params)
    end

    function self.StopEffect()

        local params = {}

        params.CHANNEL_R = gStartChannel
        params.CHANNEL_G = gStartChannel + 1
        params.CHANNEL_B = gStartChannel + 2

        if (is_rgbw() or is_rgbww()) then
            params.CHANNEL_W = gStartChannel + 3
        end

        if (is_rgbww()) then
            params.CHANNEL_WW = gStartChannel + 4
        end

        SendToProxy(self.idBinding, self.prefix.."STOP_EFFECT", params)
    end

    function self.SetAdvEffect(effect, interval, duration, colors)

        local channels = {}
        table.insert(channels, gStartChannel)
        table.insert(channels, gStartChannel + 1)
        table.insert(channels, gStartChannel + 2)

        if (is_rgbw() or is_rgbww()) then
            table.insert(channels, gStartChannel + 3)
        end

        if (is_rgbww()) then
            table.insert(channels, gStartChannel + 4)
        end

        local params = {}
        params.CHANNELS = table_to_string(channels)
        params.EFFECT = effect
        params.INTERVAL = interval
        params.DURATION = duration
        params.COLOR = table_to_string(colors)

        SendToProxy(self.idBinding, self.prefix.."SET_ADVANCED_EFFECT", params)
    end

    function self.StopAdvEffect()

        local channels = {}
        table.insert(channels, gStartChannel)
        table.insert(channels, gStartChannel + 1)
        table.insert(channels, gStartChannel + 2)

        if (is_rgbw() or is_rgbww()) then
            table.insert(channels, gStartChannel + 3)
        end

        if (is_rgbww()) then
            table.insert(channels, gStartChannel + 4)
        end

        local params = {}
        params.CHANNELS = table_to_string(channels)

        SendToProxy(self.idBinding, self.prefix.."STOP_ADVANCED_EFFECT", params)
    end

    function self.UpdateChannels()
        if (gStartChannel == "No channel selected") then return end

        local tParams = {}
        tParams.START = tonumber(gStartChannel)

        if (is_rgbww()) then
            tParams.COUNT = 5
        elseif (is_rgbw()) then
            tParams.COUNT = 4
        else
            tParams.COUNT = 3
        end

        SendToProxy(self.idBinding, self.prefix.."UPDATE_CHANNELS", tParams)
    end

    return self
end

API = ControllerInterface()

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Main
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

if (PersistData == nil) then
    PersistData = {}
end

CHANNEL_MODES = {
	"RGB",
	"RGBW",
    "RGBWW"
}

gStartChannel		   = tostring(Properties["Start Channel Selector"] - 1) or "No channel selected"
gChannelMode		   = Properties["Channel Mode Selector"] or CHANNEL_MODES[1]
gClickRampRateDown     = PersistData["ratedown"] or 50
gClickRampRateUp       = PersistData["rateup"] or 50
gHoldRampRateDown      = PersistData["holddown"] or 50
gHoldRampRateUp        = PersistData["holdup"] or 50
gPresetLevel           = PersistData["preset"] or 100
gBrightnessRateDefault = PersistData["brightnessratedefault"] or 0
gColorRateDefault      = PersistData["colorratedefault"] or 0
gDebounce              = toboolean(PersistData["debounce"] or Properties["Debounce Timer"] or false)
gDebounceUserSet       = toboolean(PersistData["debounceuserset"] or false)
gDebounceTime          = tonumber_loc(PersistData["debouncetime"] or Properties["Debounce Milliseconds"] or 500)
gDebounceIDBrightness  = nil
gDebounceIDColor       = nil

LIGHT_PROXY_BINDING 	   = 5001
DMX_CONTROLLER_BINDING	   = 1
ON_BINDING                 = 300
OFF_BINDING                = 301
TOGGLE_BINDING             = 302

function OnDriverInit()
    dbg("(OnDriverInit)")
end

function OnDriverLateInit()
    dbg("(OnDriverLateInit)")

    C4:SendToProxy(LIGHT_PROXY_BINDING, 'ONLINE_CHANGED', {STATE=true})

    OnPropertyChanged("Channel Mode Selector")
    OnPropertyChanged("Start Channel Selector")

    API.GetThemes()
 	API.GetEffects()
    API.GetAdvEffects()

    if (gStartChannel ~= "No channel selected") then
        API.UpdateChannels()
    end
end

function DebounceDefault()
    if (gDebounce == true or gDebounceUserSet == true) then return end

    local tVers = C4:GetVersionInfo()
    local major, minor, rev, build = string.match(tVers["version"], "(%d+).(%d+).(%d+).(%d+)")

    major = tonumber(major)
    minor = tonumber(minor)
    rev = tonumber(rev)

    if (major < 3) then return end
    if (major == 3) then
        if (minor < 3) then return end
        if (minor == 3 and rev < 3) then return end
    end

    dbg("(Debounce Default) Version >= 3.3.3 Detected, Enabling Debounce")
    PROPS.DebounceTimer("Enabled")
    C4:UpdateProperty("Debounce Timer", "Enabled")

    -- Reset gDebounceUserSet, since we set it
    gDebounceUserSet = false
    PersistData["debounceuserset"] = gDebounceUserSet
end
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Properties
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////


PROPS = {}

function PROPS.StartChannelSelector(channel)
	dbg("(StartChannelSelector) Channel(" .. channel .. ")")

	gStartChannel = tostring(channel - 1)

	-- Force light state change
	C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", {STATE=false})
	C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", {STATE=true})

	API.UpdateChannels()
end

function PROPS.ChannelModeSelector(mode)
	dbg("(ChannelModeSelector) Mode(" .. mode .. ")")

	RGB.SetType(mode)

	local capabilities = {
		supports_color = true,
		supports_color_correlated_temperature = true
	}
	local channels = {
		["1"] = 1,
		["2"] = 1,
		["4"] = 1,
		["5"] = 1
	}

	if (mode == "Single Color") then
		-- Set things up for Single Color
		RGB.SetWW(0)
		RGB.SetCW(0)
		capabilities.supports_color = false
		capabilities.supports_color_correlated_temperature = false
	elseif (mode == "Tunable White") then
		-- Set things up for Tunable White
		channels["1"] = 0
		channels["2"] = 0
		capabilities.supports_color = false
	elseif (mode == "RGB") then
		-- Set things up for RGB
		RGB.SetWW(0)
		RGB.SetCW(0)
	elseif (mode == "RGBW") then
		-- Set things up for RGBW
		channels["4"] = 0
		RGB.SetCW(0)
	elseif (mode == "RGBWW") then
		-- Set things up for RGBWW
		channels["4"] = 0
		channels["5"] = 0
	else
		return
	end

	C4:SendToProxy(LIGHT_PROXY_BINDING, "DYNAMIC_CAPABILITIES_CHANGED", capabilities, "NOTIFY")

	C4:SetPropertyAttribs("Channel 1 White Temp", channels["1"])
	if (channels["1"] == 0) then
		OnPropertyChanged("Channel 1 White Temp")
	end

	C4:SetPropertyAttribs("Channel 2 White Temp", channels["2"])
	if (channels["2"] == 0) then
		OnPropertyChanged("Channel 2 White Temp")
	end

	C4:SetPropertyAttribs("Channel 4 White Temp", channels["4"])
	if (channels["4"] == 0) then
		OnPropertyChanged("Channel 4 White Temp")
	end

	C4:SetPropertyAttribs("Channel 5 White Temp", channels["5"])
	if (channels["5"] == 0) then
		OnPropertyChanged("Channel 5 White Temp")
	end
end

function PROPS.Channel1WhiteTemp(temp)
	dbg("(Channel1WhiteTemp) Temp(" .. temp .. ")")

	RGB.SetWW(temp)

	C4:SendToProxy(LIGHT_PROXY_BINDING, "DYNAMIC_CAPABILITIES_CHANGED", {["color_correlated_temperature_min"] = tonumber(temp)}, "NOTIFY")
end

function PROPS.Channel2WhiteTemp(temp)
	dbg("(Channel2WhiteTemp) Temp(" .. temp .. ")")

	RGB.SetCW(temp)

	C4:SendToProxy(LIGHT_PROXY_BINDING, "DYNAMIC_CAPABILITIES_CHANGED", {["color_correlated_temperature_max"] = tonumber(temp)}, "NOTIFY")
end

function PROPS.Channel4WhiteTemp(temp)
	dbg("(Channel4WhiteTemp) Temp(" .. temp .. ")")

	RGB.SetWW(temp)

	C4:SendToProxy(LIGHT_PROXY_BINDING, "DYNAMIC_CAPABILITIES_CHANGED", {["color_correlated_temperature_min"] = tonumber(temp)}, "NOTIFY")
end

function PROPS.Channel5WhiteTemp(temp)
	dbg("(Channel5WhiteTemp) Temp(" .. temp .. ")")

	RGB.SetCW(temp)

	C4:SendToProxy(LIGHT_PROXY_BINDING, "DYNAMIC_CAPABILITIES_CHANGED", {["color_correlated_temperature_max"] = tonumber(temp)}, "NOTIFY")
end

function PROPS.DebounceTimer(state)
	dbg("(DebounceTimer) State(" .. (state or "nil") .. ")")

	if (state == "Enabled" or state == "ENABLED") then
		gDebounce = true
	else
		gDebounce = false
	end

	PersistData["debounce"] = gDebounce
	gDebounceUserSet = true
	PersistData["debounceuserset"] = gDebounceUserSet
end

function PROPS.DebounceMilliseconds(ms)
	dbg("(DebounceMilliseconds) Milliseconds(" .. (ms or "nil") .. ")")

	gDebounceTime = tonumber_loc(ms)
	PersistData["debouncetime"] = gDebounceTime
end

function COMMAND.SavePreset(tParams)
	dbg("(SavePreset) Slot("..tParams.Preset..")")

	-- 0 based index so minus 1
	local preset = tonumber_loc(tParams.Preset) - 1

	local count = gStartChannel + 2
	if (RGB.Settings.IsRGBW) then
		count = count + 1
	elseif (RGB.Settings.IsRGBWW) then
		count = count + 2
	end

	for i = gStartChannel, count do
		API.Save(i, preset)
	end
end

function COMMAND.RecallPreset(tParams)
	dbg("(RecallPreset) Slot("..tParams.Preset..")")

	-- 0 based index so minus 1
	local preset = tonumber_loc(tParams.Preset) - 1

	local count = gStartChannel + 2
	if (RGB.Settings.IsRGBW) then
		count = count + 1
	elseif (RGB.Settings.IsRGBWW) then
		count = count + 2
	end

	for i = gStartChannel, count do
		API.Recall(i, preset)
	end
end

function COMMAND.SetTheme(tParams)
	dbg("(SetTheme) Setting Theme "..tParams.Theme)

	API.SetTheme(tParams.Theme)
end

function COMMAND.SetEffect(tParams)
	dbg("(SetEffect) Setting Effect " .. tParams.Effect)

	local red, green, blue = tParams['Color Selector']:match("([^,]+),([^,]+),([^,]+)")

	red = tonumber_loc(red)
	green = tonumber_loc(green)
	blue = tonumber_loc(blue)

	-- Determine largest for brightness
	local level = red
	if (level < green) then
		level = green
	end

	if (level < blue) then
		level = blue
	end

	-- Make level 0-100
	level = math.floor((level / 255) * 100)

	-- Set up brightness before setting the color
	RGB.Settings.Current.BRIGHTNESS = tonumber(level)
	UpdateProxyLightBrightnessChanged(tonumber(level))

	-- Get XY from RGB
	local x, y = C4:ColorRGBtoXY(red, green, blue)

	-- Set color through proxy command
	local t = {
		LIGHT_COLOR_TARGET_X = tonumber(x),
		LIGHT_COLOR_TARGET_Y = tonumber(y),
		RATE = 0
	}

	SET_COLOR_TARGET(1, t)

	API.SetEffect(tParams.Effect, tParams.Interval, tParams.Duration)
end

function COMMAND.StopEffect(tParams)
	dbg("(StopEffect) Stopping current effect")

	API.StopEffect()
end

function COMMAND.SetAdvancedEffect(tParams)
	dbg("(SetAdvancedEffect) Setting Advanced Effect " .. tParams.Effect)

	local color1 = {}
	color1[tostring(gStartChannel)], color1[tostring(gStartChannel + 1)], color1[tostring(gStartChannel + 2)] = tParams['Color 1']:match("([^,]+),([^,]+),([^,]+)")

	local color2 = {}
	color2[tostring(gStartChannel)], color2[tostring(gStartChannel + 1)], color2[tostring(gStartChannel + 2)] = tParams['Color 2']:match("([^,]+),([^,]+),([^,]+)")

	local color3 = {}
	color3[tostring(gStartChannel)], color3[tostring(gStartChannel + 1)], color3[tostring(gStartChannel + 2)] = tParams['Color 3']:match("([^,]+),([^,]+),([^,]+)")

	local color4 = {}
	color4[tostring(gStartChannel)], color4[tostring(gStartChannel + 1)], color4[tostring(gStartChannel + 2)] = tParams['Color 4']:match("([^,]+),([^,]+),([^,]+)")

	local color5 = {}
	color5[tostring(gStartChannel)], color5[tostring(gStartChannel + 1)], color5[tostring(gStartChannel + 2)] = tParams['Color 5']:match("([^,]+),([^,]+),([^,]+)")

	local colors = {}
	if (check_color_table(color1)) then table.insert(colors, color1) end
	if (check_color_table(color2)) then table.insert(colors, color2) end
	if (check_color_table(color3)) then table.insert(colors, color3) end
	if (check_color_table(color4)) then table.insert(colors, color4) end
	if (check_color_table(color5)) then table.insert(colors, color5) end

	if (colors ~= {}) then
		API.SetAdvEffect(tParams.Effect, tParams.Interval, tParams.Duration, colors)
	else
		dbg("(SetAdvancedEffect) Advanced Effect's colors are empty!")
	end
end

function COMMAND.StopAdvancedEffect(tParams)
	dbg("(StopAdvancedEffect) Stopping Current Advanced Effect")

	API.StopAdvEffect()
end

function GetThemesForProgramming(currentValue, done, search, searchFilter)
	dbg('GetThemesForProgramming')
	local list = {}

	for k, v in pairs(API.themes) do
		table.insert (list, { text = tostring(v), value = tostring(v) })
	end

	return (list)
end

function GetEffectsForProgramming(currentValue, done, search, searchFilter)
	dbg('GetEffectsForProgramming')
	local list = {}

	for k, v in pairs(API.effects) do
		table.insert (list, { text = tostring(v), value = tostring(v) })
	end

	return (list)
end

function GetAdvEffectsForProgramming(currentValue, done, search, searchFilter)
	dbg('GetAdvEffectsForProgramming')
	local list = {}

	for k, v in pairs(API.adv_effects) do
		table.insert (list, { text = tostring(v), value = tostring(v) })
	end

	return (list)
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Proxy
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

PROXY_COMMANDS = PROXY_COMMANDS or {}

function ReceivedFromProxy(idBinding, strCommand, tParams)
	dbg("(ReceivedFromProxy) idBinding(" .. idBinding .. ") strCommand(" .. strCommand .. ") tParams(" .. ParamsToString(tParams) .. ")")
	tParams = tParams or {}

	if (gChannel == "No channel selected") then
		dbg("Please select a channel via the channel selector property")
		return
	end

	if (PROXY_COMMANDS[strCommand] ~= nil and type(PROXY_COMMANDS[strCommand]) == 'function') then
		PROXY_COMMANDS[strCommand](idBinding, tParams)
	elseif (BUTTON_LINKS[strCommand] ~= nil and type(BUTTON_LINKS[strCommand]) == 'function') then
		BUTTON_LINKS[strCommand](idBinding, tParams)
	else
		dbg('No Proxy Function found')
	end
end

function ParamsToString(tParams)
	if (tParams == nil) then return "nil" end
	
	if (type(tParams) == 'string') then return tParams end
	if (type(tParams) == 'table') then
		local ret = ""
		
		for k,v in pairs(tParams) do
			ret = ret .. "[" .. k .. "]: " .. ParamsToString(v) .. ", "
		end

		return ret:sub(1, -3)
	end

	return ""
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- From Light Proxy
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

BUTTON_IDS = {
    "TOP",
    "BOTTOM",
    "TOGGLE"
}

BUTTON_ACTIONS = {
    "RELEASE",
    "PRESS",
    "SINGLE CLICK",
    "DOUBLE CLICK",
    "TRIPLE CLICK"
}

function PROXY_COMMANDS.GET_CONNECTED_STATE(idBinding, tParams)
	dbg("(GET_CONNECTED_STATE)")

	C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", { STATE = true })
end

function PROXY_COMMANDS.SET_PRESET_LEVEL(idBinding, tParams)
	gPresetLevel = tParams["LEVEL"]
	PersistData["preset"] = gPresetLevel

	dbg("(SET_PRESET_LEVEL) Level("..gPresetLevel..")")
end

function PROXY_COMMANDS.SET_CLICK_RATE_DOWN(idBinding, tParams)
	local temp = tonumber(tParams.RATE)

	if (temp >= 1000) then
		temp = math.floor(temp/1000) -- convert to seconds
	else
		temp = 0
	end

	gClickRampRateDown = temp

	PersistData["ratedown"] = gClickRampRateDown

	dbg("(SET_CLICK_RATE_DOWN) Raw(" .. tParams.RATE .. ") Rate(" .. gClickRampRateDown .. " second)")
end

function PROXY_COMMANDS.SET_CLICK_RATE_UP(idBinding, tParams)
	local temp = tonumber(tParams.RATE)

	if (temp >= 1000) then
		temp = math.floor(temp/1000) -- convert to seconds
	else
		temp = 0
	end

	gClickRampRateUp = temp

	PersistData["rateup"] = gClickRampRateUp

	dbg("(SET_CLICK_RATE_UP) Raw("..tParams.RATE..") Rate(" .. gClickRampRateUp .. " second)")
end

function PROXY_COMMANDS.SET_HOLD_RATE_DOWN(idBinding, tParams)
	local temp = tonumber(tParams.RATE)

	if (temp >= 1000) then
		temp = math.floor(temp/1000) -- convert to seconds
	else
		temp = 0
	end

	gHoldRampRateDown = temp

	PersistData["holddown"] = gHoldRampRateDown

	dbg("(SET_HOLD_RATE_DOWN) Raw(" .. tParams.RATE .. ") Rate(" .. gHoldRampRateDown .. " second)")
end

function PROXY_COMMANDS.SET_HOLD_RATE_UP(idBinding, tParams)
	local temp = tonumber(tParams.RATE)

	if (temp >= 1000) then
		temp = math.floor(temp/1000) -- convert to seconds
	else
		temp = 0
	end

	gHoldRampRateUp = temp

	PersistData["holdup"] = gHoldRampRateUp

	dbg("(SET_HOLD_RATE_UP) Raw("..tParams.RATE..") Rate(" .. gHoldRampRateUp .. " second)")
end


function PROXY_COMMANDS.BUTTON_ACTION(idBinding, tParams)
	local btn = BUTTON_IDS[tonumber(tParams["BUTTON_ID"])+1]
	local act = BUTTON_ACTIONS[tonumber(tParams["ACTION"]+1)]

	dbg("(BUTTON_ACTION) BUTTON(".. btn ..") ACTION(" .. act ..")")

	if (btn == "TOP") then
	   if (act == "SINGLE CLICK") then
		  PROXY_COMMANDS.ON(LIGHT_PROXY_BINDING, tParams)
	   elseif (act == "PRESS") then
		  START_RAMP_UP()
	   elseif (act == "RELEASE") then
		  STOP_RAMPING()
	   end
	    
	elseif (btn == "BOTTOM") then
	   if (act == "SINGLE CLICK") then
		  PROXY_COMMANDS.OFF(LIGHT_PROXY_BINDING, tParams)
	   elseif (act == "PRESS") then
		  START_RAMP_DOWN()
	   elseif (act == "RELEASE") then
		  STOP_RAMPING()
	   end
	elseif (btn == "TOGGLE") then
	   if (act == "SINGLE CLICK") then
		  PROXY_COMMANDS.TOGGLE(idBinding, tParams)
	   elseif (act == "PRESS") then
		  if (gRaiseDimmer) then
			 START_RAMP_UP()
		  else
			 START_RAMP_DOWN()
		  end
	   elseif (act == "RELEASE") then
		  STOP_RAMPING()
	   end
	end
end

function PROXY_COMMANDS.ON(idBinding, tParams)
    dbg("(ON)")
    STOP_RAMPING()
    gRaiseDimmer = false
    PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(idBinding, {LIGHT_BRIGHTNESS_TARGET = gPresetLevel, RATE = (gClickRampRateUp * 1000)})
end

function PROXY_COMMANDS.OFF(idBinding, tParams)
    dbg("(OFF)")
    STOP_RAMPING()
    gRaiseDimmer = true
    PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(idBinding, {LIGHT_BRIGHTNESS_TARGET = 0, RATE = (gClickRampRateDown * 1000)})
end

function PROXY_COMMANDS.TOGGLE(idBinding, tParams)
    dbg("(TOGGLE)")
    STOP_RAMPING()
    if (RGB.Settings.Current.BRIGHTNESS > 0) then
	   gRaiseDimmer = true
	   PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(idBinding, {LIGHT_BRIGHTNESS_TARGET = 0, RATE = (gClickRampRateDown * 1000)})
    else
	   gRaiseDimmer = false
	   PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(idBinding, {LIGHT_BRIGHTNESS_TARGET = gPresetLevel, RATE = (gClickRampRateUp * 1000)})
    end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- From Light Proxy - New Color Proxies
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function PROXY_COMMANDS.ADD_DRIVER_COLOR_PRESET(idBinding, tParams)
	dbg("(ADD_DRIVER_COLOR_PRESET)")
	-- I don't think this fucking exists! "What's New" is a lie!
end

function PROXY_COMMANDS.DELETE_DRIVER_COLOR_PRESET(idBinding, tParams)
	dbg("(DELETE_DRIVER_COLOR_PRESET)")
	-- I don't think this fucking exists! "What's New" is a lie!
end

function PROXY_COMMANDS.MODIFY_DRIVER_COLOR_PRESET(idBinding, tParams)
	dbg("(MODIFY_DRIVER_COLOR_PRESET)")
	-- I don't think this fucking exists! "What's New" is a lie!
end

function PROXY_COMMANDS.UPDATE_BRIGHTNESS_ON_MODE(idBinding, tParams)
	dbg("(UPDATE_BRIGHTNESS_ON_MODE)")
end

function PROXY_COMMANDS.UPDATE_BRIGHTNESS_PRESET(idBinding, tParams)
	dbg("(UPDATE_BRIGHTNESS_PRESET)")
end

function PROXY_COMMANDS.UPDATE_COLOR_ON_MODE(idBinding, tParams)
	dbg("(UPDATE_COLOR_ON_MODE)")
end

function PROXY_COMMANDS.UPDATE_COLOR_PRESET(idBinding, tParams)
	dbg("(UPDATE_COLOR_PRESET)")
	-- I think this is the real one, but I'm not 100% on it's params.
	-- I also think this requires the SET_BRIGHTNESS_TARGET, but the
	-- doc doesn't say that was a new add. This doc sucks.

	-- Here's some real info from the driver
	--(UPDATE_COLOR_PRESET)
	--COLOR_X:  0.500592
	--COLOR_Y:  0.406651
	--ID:  2
	--NAME:  Dim Color
	--COLOR_MODE:  0
	--COMMAND:  MODIFIED

	RGB.UpdateColorPreset(tParams)
end

function PROXY_COMMANDS.UPDATE_BRIGHTNESS_RATE_DEFAULT(idBinding, tParams)
	dbg("(UPDATE_BRIGHTNESS_RATE_DEFAULT) RATE[" .. (tParams.RATE or "nil") .. "]")

	gBrightnessRateDefault = tonumber(tParams.RATE) or 0
end

function PROXY_COMMANDS.UPDATE_COLOR_RATE_DEFAULT(idBinding, tParams)
	dbg("(UPDATE_COLOR_RATE_DEFAULT) RATE[" .. (tParams.RATE or "nil") .. "]")

	gColorRateDefault = tonumber(tParams.RATE) or 0
end

function PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(idBinding, tParams)
	dbg("(SET_BRIGHTNESS_TARGET) TARGET[" .. (tParams.LIGHT_BRIGHTNESS_TARGET or "nil") .. "] RATE[" .. (tParams.LIGHT_BRIGHTNESS_TARGET_RATE or tParams.RATE or "nil") .. "]")

	-- Debounce
	if (gDebounce == true) then
		if (gDebounceIDBrightness ~= nil and gDebounceIDBrightness ~= 0) then
			dbg("(SET_BRIGHTNESS_TARGET) Debouncing previous input!")
			killTimer(gDebounceIDBrightness)
		end

		gDebounceIDBrightness = startTimer(gDebounceTime, "MILLISECONDS", function()
			SET_BRIGHTNESS_TARGET(idBinding, tParams)
		end)

		return
	end

	-- Proceed as normal
	SET_BRIGHTNESS_TARGET(idBinding, tParams)
end

function SET_BRIGHTNESS_TARGET(idBinding, tParams)
	local target = tonumber_loc(tParams.LIGHT_BRIGHTNESS_TARGET) or 0
	local rate = tonumber_loc(tParams.LIGHT_BRIGHTNESS_TARGET_RATE or tParams.RATE) or gBrightnessRateDefault
	local ramping = tParams.RAMPING or false

	local current_brightness = RGB.Settings.Current.BRIGHTNESS or 0

	local levels_per_second = ((math.abs(target - current_brightness) / 100) * 255 / (rate / 1000)) or 0

	RGB.SetBrightnessTarget(tParams)
	local r, g, b, ww, cw = RGB.GetColors()

	if (r ~= nil) then
		API.SetLevel(gStartChannel, r, levels_per_second)
	end

	if (g ~= nil) then
		API.SetLevel(gStartChannel + 1, g, levels_per_second)
	end

	if (b ~= nil) then
		API.SetLevel(gStartChannel + 2, b, levels_per_second)
	end

	if (ww ~= nil) then
		API.SetLevel(gStartChannel + 3, ww, levels_per_second)
	end

	if (cw ~= nil) then
		API.SetLevel(gStartChannel + 4, cw, levels_per_second)
	end

	if (not ramping) then 
	   UpdateProxyLightBrightnessChanging(current_brightness, target, rate)

	   startTimer(rate + 50, "MILLISECONDS", function()
		  UpdateProxyLightBrightnessChanged(target)

		  -- Update color to default if turning off
		  if (target == 0) then
			 UpdateProxyLightColorChanged(RGB.Settings.Current.X, RGB.Settings.Current.Y)
		  end
	   end)
    end
end

function PROXY_COMMANDS.SET_COLOR_TARGET(idBinding, tParams)
	dbg("(SET_COLOR_TARGET) X[" .. (tParams.LIGHT_COLOR_TARGET_X or "nil") .. "] Y[" .. (tParams.LIGHT_COLOR_TARGET_Y or "nil") .. "] RATE[" .. (tParams.LIGHT_COLOR_TARGET_RATE or tParams.RATE or "nil") .. "]")

	-- Debounce
	if (gDebounce == true) then
		if (gDebounceIDColor ~= nil and gDebounceIDColor ~= 0) then
			dbg("(SET_COLOR_TARGET) Debouncing previous input!")
			killTimer(gDebounceIDColor)
		end

		gDebounceIDColor = startTimer(gDebounceTime, "MILLISECONDS", function()
			SET_COLOR_TARGET(idBinding, tParams)
		end)

		return
	end

	-- Proceed as normal
	SET_COLOR_TARGET(idBinding, tParams)
end

function SET_COLOR_TARGET(idBinding, tParams)
	-- Get existing color levels for the rate calculation
	local r0, g0, b0, ww0, cw0 = RGB.GetColors()

	RGB.SetColorTarget(tParams)
	local r, g, b, ww, cw = RGB.GetColors()

	local rate = tonumber_loc(tParams.LIGHT_COLOR_TARGET_RATE or tParams.RATE) or gColorRateDefault

	-- Calculate levels per second
	local rLPS = 0
	if (r0 ~= nil and r ~= nil and rate ~= 0) then
		rLPS = math.abs(r0 - r) / (rate / 1000)
	end

	local gLPS = 0
	if (r0 ~= nil and r ~= nil and rate ~= 0) then
		gLPS = math.abs(g0 - g) / (rate / 1000)
	end

	local bLPS = 0
	if (b0 ~= nil and b ~= nil and rate ~= 0) then
		bLPS = math.abs(b0 - b) / (rate / 1000)
	end

	local wwLPS = 0
	if (ww0 ~= nil and ww ~= nil and rate ~= 0) then
		wwLPS = math.abs(ww0 - ww) / (rate / 1000)
	end

	local cwLPS = 0
	if (cw0 ~= nil and cw ~= nil and rate ~= 0) then
		cwLPS = math.abs(cw0 - cw) / (rate / 1000)
	end

	-- Set the levels
	if (r ~= nil) then
		API.SetLevel(gStartChannel, r, rLPS)
	end

	if (g ~= nil) then
		API.SetLevel(gStartChannel + 1, g, gLPS)
	end

	if (b ~= nil) then
		API.SetLevel(gStartChannel + 2, b, bLPS)
	end

	if (ww ~= nil) then
		API.SetLevel(gStartChannel + 3, ww, wwLPS)
	end

	if (cw ~= nil) then
		API.SetLevel(gStartChannel + 4, cw, cwLPS)
	end


	-- Doc says to send COLOR_CHANGING notify, and then a COLOR_CHANGED notify
	UpdateProxyLightColorChanging(tonumber_loc(tParams.LIGHT_COLOR_TARGET_X) or 0, tonumber_loc(tParams.LIGHT_COLOR_TARGET_Y) or 0, rate)

	startTimer(rate, "MILLISECONDS", function()
		UpdateProxyLightColorChanged(tonumber_loc(tParams.LIGHT_COLOR_TARGET_X) or 0, tonumber_loc(tParams.LIGHT_COLOR_TARGET_Y) or 0, tonumber_loc(tParams.LIGHT_COLOR_TARGET_MODE) or 0)
	end)
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- From DMX Controller
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////


function PROXY_COMMANDS.CTRL_GET_LEVEL(idBinding, tParams)
	dbg("(CTRL_GET_LEVEL) Channel("..tostring(tParams.CHANNEL)..") Level("..tostring(tParams.LEVEL)..")")
	
    if (gUpdateChannelsTimer) then killTimer(gUpdateChannelsTimer) end
    gUpdateChannelsTimer = startTimer(250, "MILLISECONDS", function()
	   API.UpdateChannels()
	end)
	

end

function PROXY_COMMANDS.CTRL_SET_LEVEL(idBinding, tParams)
	dbg("(CTRL_SET_LEVEL) Channel("..tostring(tParams.CHANNEL)..")")
end

function PROXY_COMMANDS.CTRL_STOP(idBinding, tParams)
	dbg("(CTRL_STOP) Channel("..tostring(tParams.CHANNEL)..")")
	
    if (gUpdateChannelsTimer) then killTimer(gUpdateChannelsTimer) end
    gUpdateChannelsTimer = startTimer(250, "MILLISECONDS", function()
	   API.UpdateChannels()
	end)
end

function PROXY_COMMANDS.CTRL_GET_THEMES(idBinding, tParams)
  dbg("(CTRL_GET_THEMES)")

  API.themes = {}

  for k,v in pairs(tParams) do
    table.insert(API.themes, v)
  end
end

function PROXY_COMMANDS.CTRL_GET_EFFECTS(idBinding, tParams)
  dbg("(CTRL_GET_EFFECTS)")

  API.effects = {}

  for k,v in pairs(tParams) do
    table.insert(API.effects, v)
  end
end

function PROXY_COMMANDS.CTRL_GET_ADVANCED_EFFECTS(idBinding, tParams)
  dbg("(CTRL_GET_ADVANCED_EFFECTS)")

  API.adv_effects = {}

  for k,v in pairs(tParams) do
    table.insert(API.adv_effects, v)
  end
end

gUpdateTimer = nil
function PROXY_COMMANDS.CTRL_UPDATE_CHANNELS(idBinding, tParams)
	dbg("(CTRL_UPDATE_CHANNELS)")

	if (tonumber(tParams.START) ~= tonumber(gStartChannel)) then return end

	local levels = loadstring("return " .. tParams.LEVELS)()
	if (levels == nil) then
		err("(CTRL_UPDATE_CHANNELS) Invalid levels!")
		return
	end

	local count = (#levels or 0)
	if (count <= 0 or count > 5) then
		err("(CTRL_UPDATE_CHANNELS) Invalid count!")
		return
	end

	for i = 1, #levels do
		levels[i] = tonumber(levels[i])
	end

	local delay = (tonumber(tParams.DELAY) or 0)

	local x = 0
	local y = 0
	local brightness = 0

	if (count == 3) then
		-- RGB
		x, y = C4:ColorRGBtoXY(levels[1], levels[2], levels[3])
		brightness = math.max(levels[1], levels[2], levels[3]) / 255 * 100
	elseif (count == 4) then
		-- RGBW
		local r, g, b = RGB.RGBWtoRGB(levels[1], levels[2], levels[3], levels[4])
		x, y = C4:ColorRGBtoXY(r, g, b)
		brightness = math.max(levels[1], levels[2], levels[3], levels[4]) / 255 * 100
		print("LEVELS[" .. levels[1] .. "," .. levels[2] .. "," .. levels[3] .. "," .. levels[4] .. "] -> RGB[" .. r .. "," .. g .. "," .. b .. "] -> XY[" .. x .. "," .. y .. "]")
	else
		-- RGBWW
		local r, g, b = RGB.RGBWWtoRGB(levels[1], levels[2], levels[3], levels[4], levels[5])
		x, y = C4:ColorRGBtoXY(r, g, b)
		brightness = math.max(levels[1], levels[2], levels[3], levels[4], levels[5]) / 255 * 100
	end

	brightness = math.floor(brightness)

	if (delay == 0) then
		UpdateProxyLightColorChanged(x, y)
		UpdateProxyLightBrightnessChanged(brightness)
		RGB.Settings.Current.BRIGHTNESS = tonumber(brightness)
		return
	end

	UpdateProxyLightColorChanging(x, y, delay - 250)
	UpdateProxyLightBrightnessChanging(RGB.Settings.Current.BRIGHTNESS, brightness, delay - 250)

	if (gUpdateTimer ~= nil) then
		killTimer(gUpdateTimer)
	end

	gUpdateTimer = startTimer(delay - 250, "MILLISECONDS", function()
		UpdateProxyLightColorChanged(x, y)
		UpdateProxyLightBrightnessChanged(brightness)
		RGB.Settings.Current.BRIGHTNESS = tonumber(brightness)
		gUpdateTimer = nil
	end)
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Update
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function UpdateProxy(value)
	dbg("(UpdateProxy) Level("..(value or "nil")..")")

	C4:SendToProxy(LIGHT_PROXY_BINDING, "LIGHT_LEVEL", tostring(value))
	C4:SendToProxy(LIGHT_PROXY_BINDING, "ONLINE_CHANGED", { STATE = true })

	SyncLEDS()
end

function UpdateProxyLightBrightnessChanged(current)
  dbg("(UpdateProxyLightBrightnessChanged) Current Level("..(current or "nil")..")")

  C4:SendToProxy(LIGHT_PROXY_BINDING, "LIGHT_BRIGHTNESS_CHANGED", { LIGHT_BRIGHTNESS_CURRENT = current })
  if (tonumber(current) == 0) then gRaiseDimmer = true end

  SyncLEDS()
end

function UpdateProxyLightBrightnessChanging(current, target, rate_ms)
  dbg("(UpdateProxyLightBrightnessChanging) Current Level("..(current or "nil")..") Target Level("..(target or "nil")..") Rate("..(rate_ms or "nil").." milliseconds)")

  C4:SendToProxy(LIGHT_PROXY_BINDING, "LIGHT_BRIGHTNESS_CHANGING", { LIGHT_BRIGHTNESS_CURRENT = current, LIGHT_BRIGHTNESS_TARGET = target, RATE = rate_ms })
end

function UpdateProxyLightColorChanging(tx, ty, rate)
  dbg("(UpdateProxyLightColorChanging) Target X("..(tx or "nil")..") Target Y("..(ty or "nil")..") Rate("..(rate or "nil")..")")

  local tParams = { LIGHT_COLOR_TARGET_X = tx, LIGHT_COLOR_TARGET_Y = ty, LIGHT_COLOR_TARGET_RATE = rate, RATE = rate}

  C4:SendToProxy(LIGHT_PROXY_BINDING, "LIGHT_COLOR_CHANGING", tParams)
end

function UpdateProxyLightColorChanged(x, y, mode)
  dbg("(UpdateProxyLightColorChanged) X("..(x or "nil")..") Y("..(y or "nil")..") Mode("..(mode or "nil")..")")

  local tParams = { LIGHT_COLOR_CURRENT_X = x, LIGHT_COLOR_CURRENT_Y = y }

  if (mode ~= nil) then
    tParams.LIGHT_COLOR_CURRENT_COLOR_MODE = mode
  end

  C4:SendToProxy(LIGHT_PROXY_BINDING, "LIGHT_COLOR_CHANGED", tParams)
end


--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Binding Changed
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function OnBindingChanged(idBinding, strClass, bIsBound)
    if(bIsBound) then
        dbg("(OnBindingChanged) idBinding("..idBinding..") Class("..strClass..") Bound(True)")

        if (idBinding == DMX_CONTROLLER_BINDING) then
          API.GetThemes()
          API.GetEffects()
        end
    end
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Proxy Send
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

function SendToProxy(idBinding, strCommand, tParams)
  dbg("(SendToProxy) idBinding("..idBinding..") Command("..strCommand..")")

  C4:SendToProxy(idBinding, strCommand, tParams)
end

--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
-- Keypad Functions
--//////////--//////////--//////////--//////////--//////////--//////////--//////////
--//////////--//////////--//////////--//////////--//////////--//////////--//////////

BUTTON_LINKS = {}

function BUTTON_LINKS.DO_PUSH(id, tParams)
	dbg("(DO_PUSH) id:(" .. id .. ")")

	if (id == ON_BINDING) then
	   START_RAMP_UP()
	   
    elseif (id == OFF_BINDING) then
	   START_RAMP_DOWN()
   
    elseif (id == TOGGLE_BINDING) then
	   if (gRaiseDimmer) then
		  START_RAMP_UP()
	   else
		  START_RAMP_DOWN()
	   end
    end
       
end

function BUTTON_LINKS.DO_RELEASE(id, tParams)
	dbg("(DO_RELEASE) id:(" .. id .. ")")
    STOP_RAMPING()
   
end

function BUTTON_LINKS.DO_CLICK(id, tParams)
	dbg("(DO_CLICK) id:(" .. id .. ")")
	STOP_RAMPING()
    if (id == ON_BINDING) then
        PROXY_COMMANDS.ON(LIGHT_PROXY_BINDING, tParams)
    elseif (id == OFF_BINDING) then
        PROXY_COMMANDS.OFF(LIGHT_PROXY_BINDING, tParams)
    elseif (id == TOGGLE_BINDING) then
        PROXY_COMMANDS.TOGGLE(LIGHT_PROXY_BINDING, tParams)
    end
end

function START_RAMP_UP()
    gRampingTimer = startTimer(500, "MILLISECONDS", function()
	   dbg("(START_RAMP_UP)")
	   gRamping = true
	   gRaiseDimmer = false
	   PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(idBinding, {LIGHT_BRIGHTNESS_TARGET = 100, RATE = (gHoldRampRateUp * 1000), RAMPING = true})
	   end)
end

function START_RAMP_DOWN()
    gRampingTimer = startTimer(500, "MILLISECONDS", function()
	   dbg("(START_RAMP_DOWN)")
	   gRamping = true
	   gRaiseDimmer = true
	   PROXY_COMMANDS.SET_BRIGHTNESS_TARGET(idBinding, {LIGHT_BRIGHTNESS_TARGET = 0, RATE = (gHoldRampRateDown * 1000), RAMPING = true})
	   end) 
end

function STOP_RAMPING()
    dbg("(STOP_RAMPING)")
	killTimer(gRampingTimer)
	if not (gRamping) then return end
	
	
	gRamping = false
	local r, g, b, ww, cw = RGB.GetColors()

	if (r ~= nil) then
		API.Stop(gStartChannel)
		-- API.GetLevel(gStartChannel)
	end

	if (g ~= nil) then
		API.Stop(gStartChannel + 1)
		-- API.GetLevel(gStartChannel + 1)
	end

	if (b ~= nil) then
		API.Stop(gStartChannel + 2)
		-- API.GetLevel(gStartChannel + 2)
	end

	if (ww ~= nil) then
		API.Stop(gStartChannel + 3)
		-- API.GetLevel(gStartChannel + 3)
	end

	if (cw ~= nil) then
		API.Stop(gStartChannel + 4)
		-- API.GetLevel(gStartChannel + 4)
	end
	
end

function SyncLEDS()
	dbg("(SyncLEDS) LEVEL(".. RGB.Settings.Current.BRIGHTNESS ..")")

	local led = tonumber(RGB.Settings.Current.BRIGHTNESS) > 0 and true or false

    C4:SendToProxy(OFF_BINDING, "MATCH_LED_STATE", {STATE=not led})
    C4:SendToProxy(ON_BINDING, "MATCH_LED_STATE", {STATE=led})
    C4:SendToProxy(TOGGLE_BINDING, "MATCH_LED_STATE", {STATE=led})
end

-- //
-- // RGB Handler
-- //

function RGB_Handler()
    local self = {}

    self.Settings = {
        Current = {
            X = 0,
            Y = 0,
            BRIGHTNESS = 100,
            TEMP = false
        },
        DefaultOn = {
            X = 0,
            Y = 0,
            TEMP = false
        },
        IsSingleColor = false,
        IsTunableWhite = false,
        IsRGB = true,
        IsRGBW = false,
        IsRGBWW = false,
        CoolWhiteK = 0,
        WarmWhiteK = 0,
        MiddleWhiteK = 0,
        AdjustValuesWithBrightness = true,
        Range = 255
    }

    function self.SetBrightnessTarget(tParams)
        if (tParams == nil) then return end

        self.dbg("[SetBrightnessTarget] " .. self.StringtParams(tParams))

        self.Settings.Current.BRIGHTNESS = tonumber_loc(tParams.LIGHT_BRIGHTNESS_TARGET) or 0

        -- Set to default on color if light is turned off
        if (self.Settings.Current.BRIGHTNESS <= 0) then
            self.Settings.Current.X = self.Settings.DefaultOn.X
            self.Settings.Current.Y = self.Settings.DefaultOn.Y
            self.Settings.Current.TEMP = self.Settings.DefaultOn.TEMP
        end

        self.Save()

        return self.GetColors()
    end

    function self.SetColorTarget(tParams)
        if (tParams == nil) then return end

        self.dbg("[SetColorTarget] " .. self.StringtParams(tParams))

        self.Settings.Current.X = tonumber_loc(tParams.LIGHT_COLOR_TARGET_X) or 0
        self.Settings.Current.Y = tonumber_loc(tParams.LIGHT_COLOR_TARGET_Y) or 0
        self.Settings.Current.TEMP = toboolean(tParams.LIGHT_COLOR_TARGET_MODE)

        self.Save()

        return self.GetColors()
    end

    function self.UpdateColorPreset(tParams)
        if (tParams == nil) then return end

        self.dbg("[UpdateColorPreset] " .. self.StringtParams(tParams))

        if (tParams.NAME == "On Color") then
            self.Settings.DefaultOn.X = (tonumber_loc(tParams.COLOR_X) or 0)
            self.Settings.DefaultOn.Y = (tonumber_loc(tParams.COLOR_Y) or 0)
            self.Settings.DefaultOn.TEMP = toboolean(tParams.COLOR_RATE)
        end
    end

    function self.SetType(setting)
        if (setting == nil or type(setting) ~= "string") then return end

        self.dbg("[SetType] Setting (" .. setting .. ")")

        self.Settings.IsSingleColor = false
        self.Settings.IsTunableWhite = false
        self.Settings.IsRGB = false
        self.Settings.IsRGBW = false
        self.Settings.IsRGBWW = false

        if (setting == "Single Color") then
            self.Settings.IsSingleColor = true
        elseif (setting == "Tunable White") then
            self.Settings.IsTunableWhite = true
        elseif (setting == "RGB") then
            self.Settings.IsRGB = true
        elseif (setting == "RGBW") then
            self.Settings.IsRGBW = true
        elseif (setting == "RGBWW") then
            self.Settings.IsRGBWW = true
        else
            self.err("[SetType] Unknown type: " .. setting)
        end

        self.Save()
    end

    function self.SetMiddleK(ww, cw)
        if (not self.Settings.IsRGBWW) then return end
        if (ww == nil or cw == nil) then return end
        if (ww <= 0 or cw <= 0) then return end

        self.dbg("[SetMiddleK] WW (" .. ww .. ") CW (" .. cw .. ")")

        local middle = 0

        if (self.Settings.WarmWhiteK > self.Settings.CoolWhiteK) then
            middle = (self.Settings.WarmWhiteK - self.Settings.CoolWhiteK) / 2 + self.Settings.CoolWhiteK
        else
            middle = (self.Settings.CoolWhiteK - self.Settings.WarmWhiteK) / 2 + self.Settings.WarmWhiteK
        end

        self.Settings.MiddleWhiteK = tonumber(middle) or 0

        self.Save()
    end

    function self.SetWW(kelvin)
        if (kelvin == nil) then return end

        self.dbg("[SetWW] Kelvin (" .. kelvin .. ")")

        self.Settings.WarmWhiteK = tonumber(kelvin) or 0

        self.Save()

        self.SetMiddleK(self.Settings.WarmWhiteK, self.Settings.CoolWhiteK)
    end

    function self.SetCW(kelvin)
        if (kelvin == nil) then return end

        self.dbg("[SetCW] Kelvin (" .. kelvin .. ")")

        self.Settings.CoolWhiteK = tonumber(kelvin) or 0

        self.Save()

        self.SetMiddleK(self.Settings.WarmWhiteK, self.Settings.CoolWhiteK)
    end

    function self.SetAdjustValuesWithBrightness(value)
        if (value == nil) then return end

        self.dbg("[SetAdjustValuesWithBrightness] Value (" .. tostring(value) .. ")")

        self.Settings.AdjustValuesWithBrightness = toboolean(value)

        self.Save()
    end

    function self.SetRange(range)
        if (range == nil) then return end

        self.Settings.Range = tonumber(range)

        self.Save()
    end

    function self.GetSingleColor()
        return math.floor(self.Settings.Range * (self.Settings.Current.BRIGHTNESS / 100))
    end

    function self.GetTunableWhite()
        local k = C4:ColorXYtoCCT(self.Settings.Current.X, self.Settings.Current.Y)

        local range = self.Settings.CoolWhiteK - self.Settings.WarmWhiteK
        local percent = ((k - self.Settings.WarmWhiteK) / range) * 100

        local wwperc = math.floor(100 - percent)
        local cwperc = math.floor(percent)

        local ww = 0
        local cw = 0

        if (wwperc == cwperc) then
            ww = self.Settings.Range
            cw = self.Settings.Range
        elseif (wwperc > cwperc) then
            ww = self.Settings.Range
            cw = math.floor((self.Settings.Range / wwperc) * cwperc)
        else
            ww = math.floor((self.Settings.Range / cwperc) * wwperc)
            cw = self.Settings.Range
        end

        if (self.Settings.AdjustValuesWithBrightness == true) then
            ww = math.floor(ww * (self.Settings.Current.BRIGHTNESS / 100))
            cw = math.floor(cw * (self.Settings.Current.BRIGHTNESS / 100))
        end

        return ww, cw
    end

    function self.GetRGB()
        local r, g, b = C4:ColorXYtoRGB(self.Settings.Current.X, self.Settings.Current.Y, self.Settings.Range)

        r = tonumber(r)
        g = tonumber(g)
        b = tonumber(b)

        if (self.Settings.AdjustValuesWithBrightness == true) then
            r = math.floor(r * (self.Settings.Current.BRIGHTNESS / 100))
            g = math.floor(g * (self.Settings.Current.BRIGHTNESS / 100))
            b = math.floor(b * (self.Settings.Current.BRIGHTNESS / 100))
        end

        return r, g, b
    end

    function self.GetRGBW()
        if (self.Settings.IsRGBW and self.Settings.Current.TEMP == true) then
            local k = C4:ColorXYtoCCT(self.Settings.Current.X, self.Settings.Current.Y)

            if (self.Settings.WarmWhiteK ~= 0) then
                if (k > self.Settings.WarmWhiteK - 100 and k < self.Settings.WarmWhiteK + 100) then
                    if (self.Settings.AdjustValuesWithBrightness == true) then
                        return 0, 0, 0, math.floor(self.Settings.Range * (self.Settings.Current.BRIGHTNESS / 100))
                    else
                        return 0, 0, 0, self.Settings.Range
                    end
                end
            elseif (self.Settings.CoolWhiteK ~= 0) then
                if (k > self.Settings.CoolWhiteK - 100 and k < self.Settings.CoolWhiteK + 100) then
                    if (self.Settings.AdjustValuesWithBrightness == true) then
                        return 0, 0, 0, math.floor(self.Settings.Range * (self.Settings.Current.BRIGHTNESS / 100))
                    else
                        return 0, 0, 0, self.Settings.Range
                    end
                end
            end
        end

        local r, g, b = C4:ColorXYtoRGB(self.Settings.Current.X, self.Settings.Current.Y, self.Settings.Range)

        r = tonumber(r)
        g = tonumber(g)
        b = tonumber(b)

        local w = 0

        r, g, b, w = self.RGBtoRGBW(r, g, b)

        if (self.Settings.AdjustValuesWithBrightness == true) then
            r = math.floor(r * (self.Settings.Current.BRIGHTNESS / 100))
            g = math.floor(g * (self.Settings.Current.BRIGHTNESS / 100))
            b = math.floor(b * (self.Settings.Current.BRIGHTNESS / 100))
            w = math.floor(w * (self.Settings.Current.BRIGHTNESS / 100))
        end

        return r, g, b, w
    end

    function self.GetRGBWW()

        local r = 0
        local g = 0
        local b = 0
        local ww = 0
        local cw = 0

        if (self.Settings.Current.TEMP == false) then
            r, g, b, ww = self.GetRGBW()
            return r, g, b, ww, cw
        end

        local k = C4:ColorXYtoCCT(self.Settings.Current.X, self.Settings.Current.Y)

        if (k > self.Settings.WarmWhiteK - 100 and k < self.Settings.WarmWhiteK + 100) then
            if (self.Settings.AdjustValuesWithBrightness == true) then
                ww = math.floor(self.Settings.Range * (self.Settings.Current.BRIGHTNESS / 100))
            else
                ww = self.Settings.Range
            end
        elseif (k > self.Settings.CoolWhiteK - 100 and k < self.Settings.CoolWhiteK + 100) then
            if (self.Settings.AdjustValuesWithBrightness == true) then
                cw = math.floor(self.Settings.Range * (self.Settings.Current.BRIGHTNESS / 100))
            else
                cw = self.Settings.Range
            end
        elseif (k > self.Settings.MiddleWhiteK - 100 and k < self.Settings.MiddleWhiteK + 100) then
            if (self.Settings.AdjustValuesWithBrightness == true) then
                ww = math.floor(self.Settings.Range * (self.Settings.Current.BRIGHTNESS / 100))
                cw = math.floor(self.Settings.Range * (self.Settings.Current.BRIGHTNESS / 100))
            else
                ww = self.Settings.Range
                cw = self.Settings.Range
            end
        else
            r, g, b, ww = self.GetRGBW()
            return r, g, b, ww, cw
        end

        return r, g, b, ww, cw
    end

    function self.GetColors()
        if (self.Settings.IsSingleColor == true) then
            return self.GetSingleColor()
        elseif (self.Settings.IsTunableWhite == true) then
            return self.GetTunableWhite()
        elseif (self.Settings.IsRGB == true) then
            return self.GetRGB()
        elseif (self.Settings.IsRGBW == true) then
            return self.GetRGBW()
        elseif (self.Settings.IsRGBWW == true) then
            return self.GetRGBWW()
        else
            self.err("[GetColors] Unknown device type!")
        end
    end

    function self.Load()
        if (PersistData["RGB_Handler"] ~= nil) then
            self.Settings = PersistData["RGB_Handler"]
        end
    end

    function self.Save()
        PersistData["RGB_Handler"] = self.Settings
    end

    function self.StringtParams(tParams)
        if (tParams == nil) then return "" end

        local ret = ""

        for k, v in pairs(tParams) do
            ret = ret .. tostring(k) .. "(" .. tostring(v) .. ") "
        end

        return ret
    end

    function self.RGBtoRGBW(r, g, b)

        -- //Get the maximum between R, G, and B
        tM = math.max(r, g, b)

        -- //If the maximum value is 0, immediately return pure black.
        if (tM == 0) then
            return 0, 0, 0, 0
        end

        -- //This section serves to figure out what the color with 100% hue is
        multiplier = self.Settings.Range / tM
        hueR = r * multiplier
        hueG = g * multiplier
        hueB = b * multiplier

        -- //This calculates the Whiteness (not strictly speaking Luminance) of the color
        max = math.max(hueR, hueG, hueB)
        min = math.min(hueR, hueG, hueB)
        luminance = ((max + min) / 2.0 - 127.5) * (self.Settings.Range / 127.5) / multiplier

        -- //Calculate the output values
        rO = self.Bounds(math.floor(r - luminance), 0, self.Settings.Range)
        gO = self.Bounds(math.floor(g - luminance), 0, self.Settings.Range)
        bO = self.Bounds(math.floor(b - luminance), 0, self.Settings.Range)
        wO = math.floor(luminance)

        return rO, gO, bO, wO
    end

    function self.RGBWtoRGB(r, g, b, w)
        if (w == 0) then
            -- If w is 0, then return the original rgb values
            return r, g, b
        elseif (r == 0 and g == 0 and b == 0) then
            -- If everything else is 0, then it's likely a temperature
            local x, y = C4:ColorCCTtoXY(self.Settings.WarmWhiteK)
            return C4:ColorXYtoRGB(x, y, self.Settings.Range)
        end

        local multiplier = self.Settings.Range / w
        local luminance = w + 127.5 * multiplier / self.Settings.Range

        local rO = self.Bounds(math.floor(r + luminance), 0, self.Settings.Range)
        local gO = self.Bounds(math.floor(g + luminance), 0, self.Settings.Range)
        local bO = self.Bounds(math.floor(b + luminance), 0, self.Settings.Range)

        return rO, gO, bO
    end

    function self.RGBWWtoRGB(r, g, b, ww, cw)
        if (r == 0 and g == 0 and b == 0) then
            -- If everything else is 0, then it's likely a temperature
            local temp = self.Settings.MiddleWhiteK
            if (ww == 0) then
                temp = self.Settings.CoolWhiteK
            elseif (cw == 0) then
                temp = self.Settings.WarmWhiteK
            end

            local x, y = C4:ColorCCTtoXY(temp)
            return C4:ColorXYtoRGB(x, y, self.Settings.Range)
        end

        -- I don't know what all of them on at the same time would be
        -- because the code would never do that. So I have to treat it
        -- like it's RGBW.
        return self.RGBWtoRGB(r, g, b, ww)
    end

    function self.Bounds(input, lower, upper)
        if (input < lower) then
            return lower
        elseif (input > upper) then
            return upper
        end

        return input
    end

    function self.dbg(msg)
        dbg("[RGB Handler] {DEBUG}: " .. msg)
    end

    function self.err(msg)
        print("[RGB Handler] {ERROR}: " .. msg)
    end

    return self
end

RGB = RGB_Handler()
RGB.Load()

function tonumber_loc(str, base)
    if (type(str) == "number") then return str end
    if (type(str) ~= "string") then return tonumber(str, base) end
    local s = str:gsub(",", ".") -- Assume US Locale decimal separator
    local num = tonumber(s, base)
    if (num == nil) then
        s = str:gsub("%.", ",") -- Non-US Locale decimal separator
        num = tonumber(s, base)
    end
    return num
end

function is_rgbw()
    return RGB.Settings.IsRGBW
end

function is_rgbww()
    return RGB.Settings.IsRGBWW
end

function bounds(input, lower, upper)
    if (input < lower) then
        return lower
    elseif (input > upper) then
        return upper
    end

    return input
end

function rgb_to_rgbw(r, g, b)
    -- //Get the maximum between R, G, and B
    -- float tM = Math.Max(Ri, Math.Max(Gi, Bi));
    --
    tM = math.max(r, g, b)
    -- //If the maximum value is 0, immediately return pure black.
    -- if(tM == 0)
    --    { return new rgbwcolor() { r = 0, g = 0, b = 0, w = 0 }; }
    --
    if (tM == 0) then
        return 0, 0, 0, 0
    end
    -- //This section serves to figure out what the color with 100% hue is
    -- float multiplier = 255.0f / tM;
    -- float hR = Ri * multiplier;
    -- float hG = Gi * multiplier;
    -- float hB = Bi * multiplier;
    --
    multiplier = 255.0 / tM
    hueR = r * multiplier
    hueG = g * multiplier
    hueB = b * multiplier
    -- //This calculates the Whiteness (not strictly speaking Luminance) of the color
    -- float M = Math.Max(hR, Math.Max(hG, hB));
    -- float m = Math.Min(hR, Math.Min(hG, hB));
    -- float Luminance = ((M + m) / 2.0f - 127.5f) * (255.0f/127.5f) / multiplier;
    --
    max = math.max(hueR, hueG, hueB)
    min = math.min(hueR, hueG, hueB)
    luminance = ((max + min) / 2.0 - 127.5) * (255.0 / 127.5) / multiplier
    -- //Calculate the output values
    -- int Wo = Convert.ToInt32(Luminance);
    -- int Bo = Convert.ToInt32(Bi - Luminance);
    -- int Ro = Convert.ToInt32(Ri - Luminance);
    -- int Go = Convert.ToInt32(Gi - Luminance);
    --
    rO = bounds(math.floor(r - luminance), 0, 255)
    gO = bounds(math.floor(g - luminance), 0, 255)
    bO = bounds(math.floor(b - luminance), 0, 255)
    wO = math.floor(luminance)
    -- //Trim them so that they are all between 0 and 255
    -- if (Wo < 0) Wo = 0;
    -- if (Bo < 0) Bo = 0;
    -- if (Ro < 0) Ro = 0;
    -- if (Go < 0) Go = 0;
    -- if (Wo > 255) Wo = 255;
    -- if (Bo > 255) Bo = 255;
    -- if (Ro > 255) Ro = 255;
    -- if (Go > 255) Go = 255;
    -- return new rgbwcolor() { r = Ro, g = Go, b = Bo, w = Wo };

    return rO, gO, bO, wO
end

function rgbk_to_rgbw(r, g, b, k)
    w4 = 0
    w4temp = tonumber_loc(gChannel4WhiteTemp, 10)

    if (k > w4temp - 100 and k < w4temp + 100) then
        r = 0
        g = 0
        b = 0
        w4 = 255 * (gLastBrightness / 100)
    else
        r, g, b, w4 = rgb_to_rgbw(r, g, b)
    end

    return r, g, b, w4
end

function rgbk_to_rgbww(r, g, b, k)
    w4 = 0
    w5 = 0

    w4temp = tonumber_loc(gChannel4WhiteTemp, 10)
    w5temp = tonumber_loc(gChannel5WhiteTemp, 10)

    -- Determine middle k
    middle = -1
    if (w4temp > w5temp) then
        middle = (w4temp - w5temp) / 2 + w5temp
    else
        middle = (w5temp - w4temp) / 2 + w4temp
    end

    if (k > w4temp - 100 and k < w4temp + 100) then
        r = 0
        g = 0
        b = 0
        w4 = 255 * (gLastBrightness / 100)
        w5 = 0
    elseif (k > w5temp - 100 and k < w5temp + 100) then
        r = 0
        g = 0
        b = 0
        w4 = 0
        w5 = 255 * (gLastBrightness / 100)
    elseif (k > middle - 100 and k < middle + 100) then
        r = 0
        g = 0
        b = 0
        w4 = 255 * (gLastBrightness / 100)
        w5 = 255 * (gLastBrightness / 100)
    else
        r, g, b, w4 = rgb_to_rgbw(r, g, b)
        w5 = 0
    end

    return r, g, b, w4, w5
end

function check_color_table(color)
    if (color == nil or type(color) ~= "table") then return false end

    for k,v in pairs(color) do
        if (tonumber(v) ~= 0) then return true end
    end

    return false
end

function table_to_string(tbl)
    if (tbl == nil or type(tbl) ~= "table") then return "" end

    local result = "{"
    for k, v in pairs(tbl) do
        -- Check the key type (ignore any numerical keys - assume its an array)
        if type(k) == "string" then
            result = result .. "[\"" .. k .. "\"]" .. "="
        end

        -- Check the value type
        if type(v) == "table" then
            result = result .. table_to_string(v)
        elseif type(v) == "boolean" then
            result = result .. tostring(v)
        else
            result = result .. "\"" .. v .. "\""
        end

        result = result .. ","
    end

    -- Remove trailing comma from the result
    if result ~= "{" then
        result = result:sub(1, result:len() - 1)
    end

    return result .. "}"
end

